<?php

use Yabetoo\Yabetoo;

if (!defined('ABSPATH')) {
  exit;
}
require_once(ABSPATH . 'wp-admin/includes/plugin.php');


class WC_Yabetoopay_Gateway extends WC_Payment_Gateway
{
  /**
   * Yabetoopay config to open iframe
   *
   * @var array
   */
  private array $yabetoopay_config;

  /**
   * API public key
   *
   * @var string
   */
  private $public_key;

  /**
   * API private key
   *
   * @var string
   */
  private $private_key;

  /**
   * API secret key
   *
   * @var string
   */
  private $secret;

  /**
   * Is test mode active?
   *
   * @var bool
   */
  private $testmode;

  /**
   * Gateway disabled message
   *
   * @var string
   */
  public $msg;

  public function __construct()
  {
    $this->id = 'yabetoopay-gateway';
    $this->icon = plugins_url('assets/img/yabetoopay.svg', WC_YABETOOPAY_MAIN_FILE) . '?ver=' . WC_YABETOOPAY_VERSION;
    $this->has_fields = false;
    $this->method_title = 'Yabetoopay';
    $this->description = "<div class='yabetoopay-payment-method'>
          Moov Money, MTN Money, Orange Money, TMoney, FreeMoney, Wave, Visa, Mastercard
        </div>";

    // Initialiser les champs de formulaire
    $this->init_form_fields();

    // Charger les paramètres
    $this->init_settings();

    // Récupérer les options
    $this->title = $this->get_option('title');
    $this->method_description = $this->get_option('description');
    $this->sk_test = $this->get_option('sk_test');
    $this->sk_live = $this->get_option('sk_live');
    $this->secret = $this->get_option('secret');
    $this->account_id = $this->get_option('account_id');
    $this->testmode = $this->get_option('testmode') === 'yes' ? true : false;

    $this->supports = array(
      'products',
    );

    add_action('admin_notices', array($this, 'do_ssl_check'));

    add_action('woocommerce_receipt_' . $this->id, array($this, 'receipt_page'));

    if (is_admin()) {
      add_action('woocommerce_update_options_payment_gateways_' . $this->id, array($this, 'process_admin_options'));
      //$this->import_admin_scripts();
    }

    // Payment listener/API hook.
    //add_action('woocommerce_api_wc_yabetoopay_gateway', array($this, 'on_yabetoopay_back'));

    // Webhook listener/API hook.
    //add_action('woocommerce_api_wc_yabetoopay_gateway_webhook', array($this, 'on_yabetoopay_webhook'));

    if (!$this->is_valid_for_use()) {
      $this->enabled = false;
    }
  }

  public function init_form_fields()
  {
    $this->form_fields = include(plugin_dir_path(WC_YABETOOPAY_MAIN_FILE) . 'admin/yabetoopay-settings.php');
  }

  /**
   * Check if this gateway is enabled and can work with user currency.
   */
  public function is_valid_for_use()
  {

    //verify currency
    $currency = get_woocommerce_currency();

    $allowed_currency = array('XOF', 'XAF');

    if (!in_array($currency, $allowed_currency)) {
      $this->msg = __('Yabetoopay does not support your store currency. Kindly set it to XAF (CFA)', 'yabetoopay-gateway') . '<a href="' . admin_url('admin.php?page=wc-settings&tab=general') . '">here</a>';

      return false;
    }
    return true;
  }

  // Check if we are forcing SSL on checkout pages
  public function do_ssl_check()
  {
    if ($this->enabled == "yes") {
      if (get_option('woocommerce_force_ssl_checkout') == "no") {
        echo "<div class=\"error\"><p>" . sprintf(__("<strong>%s</strong> is enabled and WooCommerce is not forcing the SSL certificate on your checkout page. Please ensure that you have a valid SSL certificate and that you are <a href=\"%s\">forcing the checkout pages to be secured.</a>", 'yabetoopay-gateway'), $this->method_title, admin_url('admin.php?page=wc-settings&tab=checkout')) . "</p></div>";
      }
    }
  }

  public function admin_options()
  {
    if (!$this->is_valid_for_use()) {
      echo '<div class="inline error"><p><strong>' . __('Yabetoopay Payment Gateway Disabled', 'yabetoopay-gateway') . '</strong>: ' . $this->msg . '</p></div>';
    } else {
      echo '<table class="form-table">';
      $this->generate_settings_html();
      echo '</table>';
    }
?>
    <h4>
      <strong><?php printf(__('Facultatif : Pour éviter les situations où un mauvais réseau rend impossible la vérification des transactions, définissez l\'URL de votre webhook <a href="%1$s" target="_blank" rel="noopener noreferrer">ici</a> sur l\'URL ci-dessous. <span><pre><code>%2$s</code></pre></span>', 'yabetoopay-gateway'), 'https://app.yabetoo.com/developers', WC()->api_request_url('Wc_Yabetoopay_Gateway_Webhook')); ?></strong>
    </h4>
<?php
  }


  public function process_payment($order_id)
  {
    $order = new WC_Order($order_id);
    $this->setYabetoopayConfig($order);
    $url = $this->request_yabetoopay_payment_session($this->yabetoopay_config);

    // Mettre la commande en attente
    return array(
      'result' => 'success',
      'redirect' => $url
    );
  }



  public function setYabetoopayConfig($order)
  {
    // Récupérer la devise
    $currency = strtolower(get_woocommerce_currency());

    // Récupérer les articles
    $items = [];
    $line_items = $order->get_items();

    foreach ($line_items as $item_id => $item) {
      $product = $item->get_product();
      $price = (int)$item->get_total() / $item['qty']; // Prix unitaire du produit

      $items[] = [
        "productId" => (string)($product ? $product->get_id() : $item["product_id"]),
        "quantity" => $item['qty'],
        "price" => (string) $price,
        "productName" => $item['name']
      ];
    }

    // Construire la configuration
    $this->yabetoopay_config = [
      "total" => (int)$order->get_total(), // Ne pas convertir en centimes
      "currency" => $currency,
      "accountId" => $this->account_id, // Utiliser la clé publique comme ID de compte
      "successUrl" => $this->get_callback_url($order->get_id()),
      "cancelUrl" => $order->get_cancel_order_url(),
      "metadata" => [
        "order_id" => $order->get_id(),
        "customer_email" => $order->get_billing_email(),
        "customer_name" => $order->get_formatted_billing_full_name()
      ],
      "items" => $items
    ];
  }

  public function get_callback_url($order_id)
  {
    return home_url('/') . '?wc-api=' . get_class($this) . '&order_id=' . $order_id;
  }

  /**
   * Handle the callback from Yabetoopay
   */
  public function handle_callback()
  {
    if (empty($_GET['order_id'])) {
      wp_die('Aucune commande spécifiée', 'Erreur de callback', array('response' => 400));
    }


    $secret_key = $this->testmode ? $this->sk_test : $this->sk_live;
    $yabetoo = new Yabetoo($secret_key);
    $intent_id = $_GET['paymentId'];

    $payment = $yabetoo->payments->retrieve($intent_id);

    if (!$payment) {
      wp_die('Paiement invalide', 'Erreur de callback', array('response' => 400));
    }

    $metadata = $payment['metadata'];
    $order_id = wc_clean($metadata['order_id']);
    $order = wc_get_order($order_id);


    if (!$order) {
      wp_die('Commande invalide', 'Erreur de callback', array('response' => 400));
    }

    $transaction_status = $payment['status'];
    $transaction_id = $payment['id'];
    $this->log('Callback reçu pour la commande ' . $order_id . ' avec statut: ' . $transaction_status);

    // Traiter selon le statut
    if ($transaction_status === 'succeeded') {
      // Paiement réussi
      $order->payment_complete($transaction_id);
      $order->add_order_note(__('Paiement Yabetoopay réussi. ID de transaction: ', 'yabetoopay-gateway') . $transaction_id);

      // Rediriger vers la page de remerciement
      wp_redirect($this->get_return_url($order));
      exit;
    } elseif ($transaction_status === 'failed' || $transaction_status === 'cancelled' || $transaction_status === 'expired') {
      // Paiement échoué
      $order->update_status('failed', __('Paiement Yabetoopay échoué.', 'yabetoopay-gateway'));

      // Rediriger vers la page de paiement échoué
      wc_add_notice(__('Paiement échoué. Veuillez réessayer.', 'yabetoopay-gateway'), 'error');
      wp_redirect(wc_get_checkout_url());
      exit;
    } else {
      // Statut inconnu ou en attente
      $order->update_status('on-hold', __('En attente de confirmation de paiement Yabetoopay.', 'yabetoopay-gateway'));

      // Rediriger vers la page d'attente
      wp_redirect($order->get_view_order_url());
      exit;
    }
  }

  /**
   * Handle the webhook from Yabetoopay
   */
  public function handle_webhook()
  {
    // Récupérer les données brutes du webhook
    // $request_body = file_get_contents('php://input');
    // $this->log('Webhook reçu: ' . $request_body);

    // // Vérifier la signature du webhook pour la sécurité
    // $headers = getallheaders();
    // $signature = isset($headers['X-Yabetoopay-Signature']) ? $headers['X-Yabetoopay-Signature'] : '';

    // if (!$this->verify_webhook_signature($request_body, $signature)) {
    //   $this->log('Signature de webhook invalide');
    //   status_header(401);
    //   exit;
    // }

    // // Décoder les données JSON
    // $data = json_decode($request_body, true);

    // if (!$data || !isset($data['metadata']) || !isset($data['metadata']['order_id'])) {
    //   $this->log('Données de webhook invalides');
    //   status_header(400);
    //   exit;
    // }

    // $order_id = $data['metadata']['order_id'];
    // $order = wc_get_order($order_id);

    // if (!$order) {
    //   $this->log('Commande invalide: ' . $order_id);
    //   status_header(400);
    //   exit;
    // }

    // // Traiter selon le statut de la transaction
    // $transaction_status = isset($data['status']) ? $data['status'] : '';
    // $transaction_id = isset($data['id']) ? $data['id'] : '';

    // if ($transaction_status === 'success' || $transaction_status === 'completed') {
    //   // Paiement réussi
    //   if (!$order->is_paid()) {
    //     $order->payment_complete($transaction_id);
    //     $order->add_order_note(__('Paiement Yabetoopay confirmé par webhook. ID de transaction: ', 'yabetoopay-woocommerce') . $transaction_id);
    //   }
    // } elseif ($transaction_status === 'failed' || $transaction_status === 'cancelled') {
    //   // Paiement échoué
    //   $order->update_status('failed', __('Paiement Yabetoopay échoué (webhook).', 'yabetoopay-woocommerce'));
    // }

    // // Répondre avec succès
    // status_header(200);
    // echo json_encode(['status' => 'success']);
    // exit;
  }

  /**
   * Vérifier la signature du webhook
   */
  private function verify_webhook_signature($payload, $signature)
  {
    // Exemple: HMAC avec votre clé secrète
    $expected_signature = hash_hmac('sha256', $payload, $this->secret);
    return hash_equals($expected_signature, $signature);
  }

  /**
   * Journaliser les messages
   */
  private function log($message)
  {
    if (defined('WP_DEBUG') && WP_DEBUG) {
      // Utiliser WC_Logger si disponible
      if (function_exists('wc_get_logger')) {
        $logger = wc_get_logger();
        $logger->debug($message, array('source' => 'yabetoopay'));
      } else {
        // Fallback sur error_log
        error_log('Yabetoopay: ' . $message);
      }
    }
  }

  public function request_yabetoopay_payment_session($config)
  {
    $secret_key = $this->testmode ? $this->sk_test : $this->sk_live;
    $yabetoo = new Yabetoo($secret_key);


    $response = $yabetoo->sessions->create($config);
    $url = $response["url"];

    return $url;
  }



  /**
   * Handle Payment Failed.
   *
   * @param WC_Order $order
   */
  public function handlePaymentFailed($order) {}
}
