<?php

class HumAi_Job_Dashboard {

	private $plugin_name;
	private $version;

	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version = $version;
	}

	public function render_dashboard_shortcode( $atts ) {
		if ( ! is_user_logged_in() ) {
			return '<p>' . __( 'You must be logged in to view this page.', 'humai-job-posting' ) . '</p>';
		}

		if ( ! current_user_can( 'employer' ) && ! current_user_can( 'administrator' ) ) {
			return '<p>' . __( 'You do not have permission to view this page.', 'humai-job-posting' ) . '</p>';
		}

		$action = isset( $_GET['job_action'] ) ? sanitize_text_field( $_GET['job_action'] ) : 'list';
		$job_id = isset( $_GET['job_id'] ) ? intval( $_GET['job_id'] ) : 0;

		ob_start();

		if ( 'edit' === $action || 'add' === $action ) {
			$this->render_job_form( $job_id );
		} else {
			$this->render_job_list();
		}

		return ob_get_clean();
	}

	private function render_job_list() {
		$args = array(
			'post_type'      => 'job_listing',
			'post_status'    => array( 'publish', 'pending', 'draft' ),
			'author'         => get_current_user_id(),
			'posts_per_page' => -1,
		);

		$query = new WP_Query( $args );

		echo '<div class="humai-dashboard-header">';
		echo '<h2>' . __( 'My Jobs', 'humai-job-posting' ) . '</h2>';
		echo '<a href="' . esc_url( add_query_arg( 'job_action', 'add' ) ) . '" class="humai-btn">' . __( 'Post a Job', 'humai-job-posting' ) . '</a>';
		echo '</div>';

		if ( $query->have_posts() ) {
			echo '<table class="humai-dashboard-table">';
			echo '<thead><tr><th>' . __( 'Title', 'humai-job-posting' ) . '</th><th>' . __( 'Status', 'humai-job-posting' ) . '</th><th>' . __( 'Views', 'humai-job-posting' ) . '</th><th>' . __( 'Applications', 'humai-job-posting' ) . '</th><th>' . __( 'Date', 'humai-job-posting' ) . '</th><th>' . __( 'Actions', 'humai-job-posting' ) . '</th></tr></thead>';
			echo '<tbody>';
			while ( $query->have_posts() ) {
				$query->the_post();
				$job_id = get_the_ID();
				$edit_url = add_query_arg( array( 'job_action' => 'edit', 'job_id' => $job_id ) );
				$delete_url = wp_nonce_url( add_query_arg( array( 'action' => 'humai_delete_job', 'job_id' => $job_id ), admin_url( 'admin-post.php' ) ), 'humai_delete_job_' . $job_id );
				
				$views = get_post_meta( $job_id, '_humai_job_views', true );
				$views = $views ? $views : 0;

				$app_query = new WP_Query( array(
					'post_type' => 'job_application',
					'meta_key' => '_job_applied_for',
					'meta_value' => $job_id,
					'fields' => 'ids',
				) );
				$app_count = $app_query->found_posts;

				echo '<tr>';
				echo '<td>' . get_the_title() . '</td>';
				echo '<td>' . get_post_status_object( get_post_status() )->label . '</td>';
				echo '<td>' . esc_html( $views ) . '</td>';
				echo '<td>' . esc_html( $app_count ) . '</td>';
				echo '<td>' . get_the_date() . '</td>';
				echo '<td>';
				echo '<a href="' . esc_url( $edit_url ) . '">' . __( 'Edit', 'humai-job-posting' ) . '</a> | ';
				echo '<a href="' . esc_url( $delete_url ) . '" onclick="return confirm(\'' . __( 'Are you sure?', 'humai-job-posting' ) . '\');" class="humai-delete-link">' . __( 'Delete', 'humai-job-posting' ) . '</a>';
				echo '</td>';
				echo '</tr>';
			}
			echo '</tbody>';
			echo '</table>';
			wp_reset_postdata();
		} else {
			echo '<p>' . __( 'You have not posted any jobs yet.', 'humai-job-posting' ) . '</p>';
		}
	}

	private function render_job_form( $job_id = 0 ) {
		$title = '';
		$content = '';
		$location = '';
		$salary = '';
		$deadline = '';
		$email = '';
		$job_cat = array();
		$job_type = array();

		if ( $job_id ) {
			$post = get_post( $job_id );
			if ( $post && $post->post_author == get_current_user_id() ) {
				$title = $post->post_title;
				$content = $post->post_content;
				$location = get_post_meta( $job_id, '_humai_job_location', true );
				$salary = get_post_meta( $job_id, '_humai_job_salary', true );
				$deadline = get_post_meta( $job_id, '_humai_job_deadline', true );
				$email = get_post_meta( $job_id, '_humai_job_application_email', true );
				$job_cat = wp_get_object_terms( $job_id, 'job_category', array( 'fields' => 'ids' ) );
				$job_type = wp_get_object_terms( $job_id, 'job_type', array( 'fields' => 'ids' ) );
			}
		}

		echo '<h3>' . ( $job_id ? __( 'Edit Job', 'humai-job-posting' ) : __( 'Post a Job', 'humai-job-posting' ) ) . '</h3>';
		?>
		<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" class="humai-job-form">
			<input type="hidden" name="action" value="humai_save_job">
			<input type="hidden" name="job_id" value="<?php echo esc_attr( $job_id ); ?>">
			<?php wp_nonce_field( 'humai_save_job', 'humai_job_nonce' ); ?>

			<p>
				<label><?php _e( 'Job Title', 'humai-job-posting' ); ?></label>
				<input type="text" name="job_title" value="<?php echo esc_attr( $title ); ?>" required>
			</p>

			<p>
				<label><?php _e( 'Description', 'humai-job-posting' ); ?></label>
				<?php wp_editor( $content, 'job_description', array( 'textarea_name' => 'job_description', 'media_buttons' => false, 'textarea_rows' => 10 ) ); ?>
			</p>

			<p>
				<label><?php _e( 'Location', 'humai-job-posting' ); ?></label>
				<input type="text" name="job_location" value="<?php echo esc_attr( $location ); ?>">
			</p>

			<p>
				<label><?php _e( 'Salary', 'humai-job-posting' ); ?></label>
				<input type="text" name="job_salary" value="<?php echo esc_attr( $salary ); ?>">
			</p>

			<p>
				<label><?php _e( 'Application Deadline', 'humai-job-posting' ); ?></label>
				<input type="date" name="job_deadline" value="<?php echo esc_attr( $deadline ); ?>">
			</p>

			<p>
				<label><?php _e( 'Application Email', 'humai-job-posting' ); ?></label>
				<input type="email" name="job_application_email" value="<?php echo esc_attr( $email ); ?>">
			</p>
			
			<!-- Categories and Types would go here, simplified for brevity -->

			<p>
				<button type="submit" class="humai-btn"><?php _e( 'Save Job', 'humai-job-posting' ); ?></button>
				<a href="<?php echo remove_query_arg( array( 'job_action', 'job_id' ) ); ?>" class="humai-cancel-link"><?php _e( 'Cancel', 'humai-job-posting' ); ?></a>
			</p>
		</form>
		<?php
	}

	public function handle_save_job() {
		if ( ! isset( $_POST['humai_job_nonce'] ) || ! wp_verify_nonce( $_POST['humai_job_nonce'], 'humai_save_job' ) ) {
			wp_die( 'Security check failed' );
		}

		if ( ! is_user_logged_in() ) {
			wp_die( 'You must be logged in' );
		}

		$job_id = isset( $_POST['job_id'] ) ? intval( $_POST['job_id'] ) : 0;
		$title = sanitize_text_field( $_POST['job_title'] );
		$content = wp_kses_post( $_POST['job_description'] );

		$post_data = array(
			'post_title'   => $title,
			'post_content' => $content,
			'post_type'    => 'job_listing',
			'post_status'  => 'publish', // Or pending based on settings
		);

		if ( $job_id ) {
			$post = get_post( $job_id );
			if ( $post->post_author != get_current_user_id() && ! current_user_can( 'administrator' ) ) {
				wp_die( 'Unauthorized' );
			}
			$post_data['ID'] = $job_id;
			wp_update_post( $post_data );
		} else {
			$job_id = wp_insert_post( $post_data );
		}

		if ( $job_id ) {
			update_post_meta( $job_id, '_humai_job_location', sanitize_text_field( $_POST['job_location'] ) );
			update_post_meta( $job_id, '_humai_job_salary', sanitize_text_field( $_POST['job_salary'] ) );
			update_post_meta( $job_id, '_humai_job_deadline', sanitize_text_field( $_POST['job_deadline'] ) );
			update_post_meta( $job_id, '_humai_job_application_email', sanitize_email( $_POST['job_application_email'] ) );
		}

		wp_redirect( remove_query_arg( array( 'job_action', 'job_id' ), wp_get_referer() ) );
		exit;
	}
	
	public function handle_delete_job() {
		$job_id = isset( $_GET['job_id'] ) ? intval( $_GET['job_id'] ) : 0;
		check_admin_referer( 'humai_delete_job_' . $job_id );

		if ( ! is_user_logged_in() ) {
			wp_die( 'You must be logged in' );
		}

		$post = get_post( $job_id );
		if ( $post && ( $post->post_author == get_current_user_id() || current_user_can( 'administrator' ) ) ) {
			wp_trash_post( $job_id );
		}

		wp_redirect( remove_query_arg( array( 'action', 'job_id', '_wpnonce' ), wp_get_referer() ) );
		exit;
	}

}
