<?php

class HumAi_Job_Public {

	private $plugin_name;
	private $version;

	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version = $version;
	}

	public function enqueue_styles() {
		wp_enqueue_style( $this->plugin_name, plugin_dir_url( dirname( __FILE__, 2 ) ) . 'assets/css/humai-job-public.css', array(), $this->version, 'all' );
	}

	public function enqueue_scripts() {
		wp_enqueue_script( $this->plugin_name, plugin_dir_url( dirname( __FILE__, 2 ) ) . 'assets/js/humai-job-public.js', array( 'jquery' ), $this->version, false );
	}

	public function render_jobs_shortcode( $atts ) {
		$atts = shortcode_atts( array(
			'posts_per_page' => 10,
		), $atts, 'humai_jobs' );

		$paged = ( get_query_var( 'paged' ) ) ? get_query_var( 'paged' ) : 1;

		$args = array(
			'post_type'      => 'job_listing',
			'posts_per_page' => $atts['posts_per_page'],
			'post_status'    => 'publish',
			'paged'          => $paged,
		);

		// Filter by Keywords
		if ( ! empty( $_GET['search_keywords'] ) ) {
			$args['s'] = sanitize_text_field( $_GET['search_keywords'] );
		}

		// Filter by Location (Meta Query)
		if ( ! empty( $_GET['search_location'] ) ) {
			$args['meta_query'][] = array(
				'key'     => '_humai_job_location',
				'value'   => sanitize_text_field( $_GET['search_location'] ),
				'compare' => 'LIKE',
			);
		}

		// Filter by Category
		if ( ! empty( $_GET['search_category'] ) ) {
			$args['tax_query'][] = array(
				'taxonomy' => 'job_category',
				'field'    => 'slug',
				'terms'    => sanitize_text_field( $_GET['search_category'] ),
			);
		}

		// Filter by Job Type
		if ( ! empty( $_GET['search_type'] ) ) {
			$args['tax_query'][] = array(
				'taxonomy' => 'job_type',
				'field'    => 'slug',
				'terms'    => sanitize_text_field( $_GET['search_type'] ),
			);
		}

		$query = new WP_Query( $args );
		
		ob_start();
		
		echo $this->render_search_form();

		if ( $query->have_posts() ) {
			echo '<div class="humai-job-listings">';
			while ( $query->have_posts() ) {
				$query->the_post();
				$this->load_template( 'content-job-listing' );
			}
			echo '</div>';
			
			// Pagination
			echo '<div class="humai-pagination">';
			echo paginate_links( array(
				'total' => $query->max_num_pages,
			) );
			echo '</div>';

			wp_reset_postdata();
		} else {
			echo '<p>' . __( 'No jobs found.', 'humai-job-posting' ) . '</p>';
		}

		return ob_get_clean();
	}

	private function render_search_form() {
		$keywords = isset( $_GET['search_keywords'] ) ? sanitize_text_field( $_GET['search_keywords'] ) : '';
		$location = isset( $_GET['search_location'] ) ? sanitize_text_field( $_GET['search_location'] ) : '';
		$category = isset( $_GET['search_category'] ) ? sanitize_text_field( $_GET['search_category'] ) : '';
		$type     = isset( $_GET['search_type'] ) ? sanitize_text_field( $_GET['search_type'] ) : '';

		$categories = get_terms( array( 'taxonomy' => 'job_category', 'hide_empty' => false ) );
		$types      = get_terms( array( 'taxonomy' => 'job_type', 'hide_empty' => false ) );

		ob_start();
		?>
		<form method="GET" class="humai-job-search-form">
			<input type="text" name="search_keywords" placeholder="<?php _e( 'Keywords', 'humai-job-posting' ); ?>" value="<?php echo esc_attr( $keywords ); ?>">
			<input type="text" name="search_location" placeholder="<?php _e( 'Location', 'humai-job-posting' ); ?>" value="<?php echo esc_attr( $location ); ?>">
			
			<select name="search_category">
				<option value=""><?php _e( 'All Categories', 'humai-job-posting' ); ?></option>
				<?php foreach ( $categories as $cat ) : ?>
					<option value="<?php echo esc_attr( $cat->slug ); ?>" <?php selected( $category, $cat->slug ); ?>><?php echo esc_html( $cat->name ); ?></option>
				<?php endforeach; ?>
			</select>

			<select name="search_type">
				<option value=""><?php _e( 'All Job Types', 'humai-job-posting' ); ?></option>
				<?php foreach ( $types as $t ) : ?>
					<option value="<?php echo esc_attr( $t->slug ); ?>" <?php selected( $type, $t->slug ); ?>><?php echo esc_html( $t->name ); ?></option>
				<?php endforeach; ?>
			</select>

			<button type="submit"><?php _e( 'Search', 'humai-job-posting' ); ?></button>
		</form>
		<?php
		return ob_get_clean();
	}

	public function load_single_template( $template ) {
		if ( is_singular( 'job_listing' ) ) {
			// Increment View Count
			$job_id = get_the_ID();
			$views = get_post_meta( $job_id, '_humai_job_views', true );
			$views = $views ? intval( $views ) + 1 : 1;
			update_post_meta( $job_id, '_humai_job_views', $views );

			$plugin_template = plugin_dir_path( dirname( __FILE__, 2 ) ) . 'templates/single-job.php';
			if ( file_exists( $plugin_template ) ) {
				return $plugin_template;
			}
		}
		return $template;
	}

	public static function get_application_fields( $job_id ) {
		// Try to fetch from Form Editor options
		$saved_fields = get_option( 'humai_job_form_settings' );
		
		if ( ! empty( $saved_fields ) && is_array( $saved_fields ) ) {
			$fields = $saved_fields;
		} else {
			// Defaults
			$fields = array(
				'applicant_name' => array(
					'label'       => __( 'Full Name', 'humai-job-posting' ),
					'type'        => 'text',
					'required'    => true,
					'placeholder' => '',
				),
				'applicant_email' => array(
					'label'       => __( 'Email Address', 'humai-job-posting' ),
					'type'        => 'email',
					'required'    => true,
					'placeholder' => '',
				),
				'applicant_message' => array(
					'label'       => __( 'Cover Letter / Message', 'humai-job-posting' ),
					'type'        => 'textarea',
					'required'    => false,
					'placeholder' => '',
				),
				'applicant_cv' => array(
					'label'       => __( 'Upload CV/Resume (PDF, Doc)', 'humai-job-posting' ),
					'type'        => 'file',
					'required'    => true,
					'accept'      => '.pdf,.doc,.docx',
				),
			);
		}

		return apply_filters( 'humai_job_application_fields', $fields, $job_id );
	}

	public function handle_application_submission() {
		// Spam Protection (Honeypot)
		if ( ! empty( $_POST['humai_honeypot'] ) ) {
			wp_die( __( 'Spam detected.', 'humai-job-posting' ) );
		}

		if ( ! isset( $_POST['humai_application_nonce'] ) || ! wp_verify_nonce( $_POST['humai_application_nonce'], 'humai_application_submit' ) ) {
			wp_die( __( 'Security check failed', 'humai-job-posting' ) );
		}

		$job_id = isset( $_POST['job_id'] ) ? intval( $_POST['job_id'] ) : 0;
		if ( ! $job_id ) {
			wp_die( __( 'Invalid Job ID.', 'humai-job-posting' ) );
		}

		$fields = self::get_application_fields( $job_id );
		$data = array();
		$cv_url = '';

		// Validate & Sanitize
		foreach ( $fields as $key => $field ) {
			if ( ! empty( $field['required'] ) ) {
				if ( $field['type'] === 'file' ) {
					if ( empty( $_FILES[ $key ]['name'] ) ) {
						wp_die( sprintf( __( '%s is required.', 'humai-job-posting' ), $field['label'] ) );
					}
				} else {
					if ( empty( $_POST[ $key ] ) ) {
						wp_die( sprintf( __( '%s is required.', 'humai-job-posting' ), $field['label'] ) );
					}
				}
			}

			if ( $field['type'] === 'text' || $field['type'] === 'select' ) {
				$data[ $key ] = isset( $_POST[ $key ] ) ? sanitize_text_field( $_POST[ $key ] ) : '';
			} elseif ( $field['type'] === 'email' ) {
				$data[ $key ] = isset( $_POST[ $key ] ) ? sanitize_email( $_POST[ $key ] ) : '';
			} elseif ( $field['type'] === 'textarea' ) {
				$data[ $key ] = isset( $_POST[ $key ] ) ? sanitize_textarea_field( $_POST[ $key ] ) : '';
			} elseif ( $field['type'] === 'file' && ! empty( $_FILES[ $key ]['name'] ) ) {
				// Handle File Upload - currently optimized for 'applicant_cv' but could be genericized
				require_once( ABSPATH . 'wp-admin/includes/file.php' );
				$uploadedfile = $_FILES[ $key ];
				$upload_overrides = array( 'test_form' => false );
				$movefile = wp_handle_upload( $uploadedfile, $upload_overrides );

				if ( $movefile && ! isset( $movefile['error'] ) ) {
					$url = $movefile['url'];
					if ( $key === 'applicant_cv' ) {
						$cv_url = $url;
					}
					$data[ $key ] = $url;
				} else {
					wp_die( $movefile['error'] );
				}
			}
		}

		$name    = isset( $data['applicant_name'] ) ? $data['applicant_name'] : __( 'Anonymous', 'humai-job-posting' );
		$email   = isset( $data['applicant_email'] ) ? $data['applicant_email'] : '';
		$message = isset( $data['applicant_message'] ) ? $data['applicant_message'] : '';

		// Create Application Post
		$post_data = array(
			'post_title'   => sprintf( __( 'Application for %s by %s', 'humai-job-posting' ), get_the_title( $job_id ), $name ),
			'post_content' => $message,
			'post_status'  => 'publish',
			'post_type'    => 'job_application',
		);

		$application_id = wp_insert_post( $post_data );

		if ( $application_id ) {
			update_post_meta( $application_id, '_job_applied_for', $job_id );
			update_post_meta( $application_id, '_applicant_name', $name );
			update_post_meta( $application_id, '_applicant_email', $email );
			if ( $cv_url ) {
				update_post_meta( $application_id, '_applicant_cv', $cv_url );
			}
			
			// Save full data array for custom fields
			update_post_meta( $application_id, '_humai_application_data', $data );

			// Send Email Notification
			$to = get_post_meta( $job_id, '_humai_job_application_email', true );
			if ( ! $to ) {
				$to = get_option( 'admin_email' );
			}
			
			$admin_email = get_option( 'admin_email' );
			$headers = array( 'Content-Type: text/plain; charset=UTF-8' );
			
			// Add BCC to Admin
			if ( $to !== $admin_email ) {
				$headers[] = 'Bcc: ' . $admin_email;
			}

			$subject = sprintf( __( 'New Application: %s', 'humai-job-posting' ), get_the_title( $job_id ) );
			
			$body = sprintf( __( "New application received for: %s\n\n", 'humai-job-posting' ), get_the_title( $job_id ) );
			foreach ( $fields as $key => $field ) {
				if ( $field['type'] !== 'file' && isset( $data[ $key ] ) ) {
					$body .= $field['label'] . ': ' . $data[ $key ] . "\n";
				} elseif ( $field['type'] === 'file' && isset( $data[ $key ] ) ) {
					$body .= $field['label'] . ': ' . $data[ $key ] . "\n";
				}
			}

			wp_mail( $to, $subject, $body, $headers );

			// Redirect back to job page with success message
			wp_redirect( add_query_arg( 'application_sent', 'true', get_permalink( $job_id ) ) );
			exit;
		} else {
			wp_die( __( 'Error saving application.', 'humai-job-posting' ) );
		}
	}

	private function load_template( $template_name ) {
		$template_path = plugin_dir_path( dirname( __FILE__, 2 ) ) . 'templates/' . $template_name . '.php';
		if ( file_exists( $template_path ) ) {
			include $template_path;
		} else {
			// Fallback content if template missing
			echo '<div class="humai-job-item">';
			echo '<h3><a href="' . get_permalink() . '">' . get_the_title() . '</a></h3>';
			echo '<div class="humai-job-meta">';
			echo '<span>' . get_post_meta( get_the_ID(), '_humai_job_location', true ) . '</span>';
			echo ' | ';
			echo '<span>' . get_the_term_list( get_the_ID(), 'job_type', '', ', ' ) . '</span>';
			echo '</div>';
			echo '</div>';
		}
	}

}


