<?php

class Humera_Job_Admin {

	private $plugin_name;
	private $version;

	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version = $version;
	}

	public function enqueue_styles() {
		wp_enqueue_style( $this->plugin_name, HUMERA_JOB_BOARD_URL . 'assets/css/humera-job-admin.css', array(), $this->version, 'all' );
	}

	public function enqueue_scripts() {
		wp_enqueue_script( $this->plugin_name, HUMERA_JOB_BOARD_URL . 'assets/js/humera-job-admin.js', array( 'jquery' ), $this->version, false );

        // Enqueue Form Editor Script only on the specific page
        if ( isset( $_GET['page'] ) && 'humera-job-form-editor' === $_GET['page'] ) {
            wp_enqueue_script( 'humera-job-form-editor', HUMERA_JOB_BOARD_URL . 'assets/js/admin-form-editor.js', array( 'jquery' ), $this->version, true );
            
            // Localize script to pass PHP data
            $saved_fields = get_option( 'humera_job_form_settings', array() );
            // Default fields count if empty
            if ( empty( $saved_fields ) ) {
                $count = 4; // Default validation count based on default array size in display_form_editor_page
            } else {
                $count = count( $saved_fields );
            }
            
            wp_localize_script( 'humera-job-form-editor', 'humeraJobAdmin', array(
                'fieldIndex' => $count
            ) );
        }
	}

	public function display_plugin_setup_page() {
		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'Humera Job Board - Help & Shortcodes', 'humera-job-board' ); ?></h1>
			
			<div class="card">
				<h2><?php esc_html_e( 'Available Shortcodes', 'humera-job-board' ); ?></h2>
				
				<table class="form-table" role="presentation">
					<tbody>
						<tr>
							<th scope="row"><code>[humera_jobs]</code></th>
							<td>
								<p><?php esc_html_e( 'Displays the list of latest job postings with a search form.', 'humera-job-board' ); ?></p>
								<p><strong><?php esc_html_e( 'Attributes:', 'humera-job-board' ); ?></strong></p>
								<ul>
									<li><code>posts_per_page="10"</code> - <?php esc_html_e( 'Number of jobs to show per page (default: 10).', 'humera-job-board' ); ?></li>
								</ul>
								<p><em><?php esc_html_e( 'Example:', 'humera-job-board' ); ?></em> <code>[humera_jobs posts_per_page="5"]</code></p>
							</td>
						</tr>
						<tr>
							<th scope="row"><code>[humera_employer_dashboard]</code></th>
							<td>
								<p><?php esc_html_e( 'Displays the frontend dashboard for employers to manage their posted jobs.', 'humera-job-board' ); ?></p>
							</td>
						</tr>
					</tbody>
				</table>
			</div>

			<div class="card">
				<h2><?php esc_html_e( 'Customization Guide', 'humera-job-board' ); ?></h2>
				<p><?php _e( 'You can customize the application form fields using the <code>humera_job_application_fields</code> filter in your theme functions.', 'humera-job-board' ); ?></p>
			</div>
		</div>
		<?php
	}

	public function add_plugin_admin_menu() {
		add_submenu_page(
			'edit.php?post_type=job_listing',
			__( 'Help & Shortcodes', 'humera-job-board' ),
			__( 'Help & Shortcodes', 'humera-job-board' ),
			'manage_options',
			'humera-job-help',
			array( $this, 'display_plugin_setup_page' )
		);
		
		add_submenu_page(
			'edit.php?post_type=job_listing',
			__( 'Form Editor', 'humera-job-board' ),
			__( 'Form Editor', 'humera-job-board' ),
			'manage_options',
			'humera-job-form-editor',
			array( $this, 'display_form_editor_page' )
		);
	}

	public function display_form_editor_page() {
		// Handle Save
		if ( isset( $_POST['humera_save_form_fields'] ) && check_admin_referer( 'humera_save_form_fields_nonce', 'humera_save_form_fields' ) ) {
			if ( ! empty( $_POST['fields'] ) ) {
				$fields = isset( $_POST['fields'] ) ? wp_unslash( $_POST['fields'] ) : array();
				$sanitized_fields = array();
				
				if ( is_array( $fields ) ) {
					foreach ( $fields as $key => $field ) {
						// Use specific keys enabled in the UI
						if ( ! empty( $field['key'] ) ) {
							$field_key = sanitize_title( $field['key'] );
							$sanitized_fields[ $field_key ] = array(
								'label'    => sanitize_text_field( $field['label'] ),
								'type'     => sanitize_text_field( $field['type'] ),
								'required' => isset( $field['required'] ),
							);
						}
					}
					update_option( 'humera_job_form_settings', $sanitized_fields );
					echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Form fields saved successfully.', 'humera-job-board' ) . '</p></div>';
				}
			}
		}

		// Get current fields or defaults
		$saved_fields = get_option( 'humera_job_form_settings', array() );
		
		// If empty, populate with defaults for display
		if ( empty( $saved_fields ) ) {
			$saved_fields = array(
				'applicant_name' => array( 'label' => 'Full Name', 'type' => 'text', 'required' => true ),
				'applicant_email' => array( 'label' => 'Email Address', 'type' => 'email', 'required' => true ),
				'applicant_message' => array( 'label' => 'Cover Letter / Message', 'type' => 'textarea', 'required' => false ),
				'applicant_cv' => array( 'label' => 'Upload CV', 'type' => 'file', 'required' => true ),
			);
		}
		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'Application Form Editor', 'humera-job-board' ); ?></h1>
			<p><?php esc_html_e( 'Customize the fields that appear on the job application form.', 'humera-job-board' ); ?></p>
			
			<form method="post" action="">
				<?php wp_nonce_field( 'humera_save_form_fields_nonce', 'humera_save_form_fields' ); ?>
				
				<table class="widefat fixed striped" id="humera-fields-table">
					<thead>
						<tr>
							<th style="width: 20%;"><?php esc_html_e( 'Field Key (Unique)', 'humera-job-board' ); ?></th>
							<th style="width: 30%;"><?php esc_html_e( 'Label', 'humera-job-board' ); ?></th>
							<th style="width: 20%;"><?php esc_html_e( 'Type', 'humera-job-board' ); ?></th>
							<th style="width: 10%;"><?php esc_html_e( 'Required', 'humera-job-board' ); ?></th>
							<th style="width: 10%;"><?php esc_html_e( 'Actions', 'humera-job-board' ); ?></th>
						</tr>
					</thead>
					<tbody id="humera-fields-body">
						<?php 
						$index = 0;
						foreach ( $saved_fields as $key => $field ) : 
						?>
						<tr class="field-row">
							<td><input type="text" name="fields[<?php echo esc_attr( $index ); ?>][key]" value="<?php echo esc_attr( $key ); ?>" class="regular-text" required ></td>
							<td><input type="text" name="fields[<?php echo esc_attr( $index ); ?>][label]" value="<?php echo esc_attr( $field['label'] ); ?>" class="regular-text" required></td>
							<td>
								<select name="fields[<?php echo esc_attr( $index ); ?>][type]">
									<option value="text" <?php selected( $field['type'], 'text' ); ?>>Text</option>
									<option value="email" <?php selected( $field['type'], 'email' ); ?>>Email</option>
									<option value="textarea" <?php selected( $field['type'], 'textarea' ); ?>>Textarea</option>
									<option value="file" <?php selected( $field['type'], 'file' ); ?>>File Select</option>
								</select>
							</td>
							<td><input type="checkbox" name="fields[<?php echo esc_attr( $index ); ?>][required]" <?php checked( ! empty( $field['required'] ) ); ?>></td>
							<td><button type="button" class="button humera-remove-field">Remove</button></td>
						</tr>
						<?php 
						$index++;
						endforeach; 
						?>
					</tbody>
				</table>
				
				<p>
					<button type="button" class="button" id="humera-add-field"><?php esc_html_e( '+ Add Field', 'humera-job-board' ); ?></button>
				</p>
				
				<hr>
				<p>
					<button type="submit" class="button button-primary"><?php esc_html_e( 'Save Form Fields', 'humera-job-board' ); ?></button>
				</p>
			</form>
		</div>
		<?php
	}
	
	public function add_meta_boxes() {
		add_meta_box(
			'humera_job_application_actions',
			__( 'Actions', 'humera-job-board' ),
			array( $this, 'render_application_actions_meta_box' ),
			'job_application',
			'side',
			'high'
		);
	}
	
	public function render_application_actions_meta_box( $post ) {
		$print_url = admin_url( 'admin-post.php?action=humera_print_application&post_id=' . $post->ID );
		?>
		<div class="misc-pub-section">
			<a href="<?php echo esc_url( $print_url ); ?>" target="_blank" class="button button-primary button-large" style="width: 100%; text-align: center;">
				<?php esc_html_e( 'Download / Print Application', 'humera-job-board' ); ?>
			</a>
		</div>
		<?php
	}
	
	public function handle_print_application() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Unauthorized access.', 'humera-job-board' ) );
		}
		
		$post_id = isset( $_GET['post_id'] ) ? intval( $_GET['post_id'] ) : 0;
		if ( ! $post_id ) {
			wp_die( esc_html__( 'Invalid Post ID.', 'humera-job-board' ) );
		}
		
		$post = get_post( $post_id );
		if ( ! $post || $post->post_type !== 'job_application' ) {
			wp_die( esc_html__( 'Application not found.', 'humera-job-board' ) );
		}
		
		$applicant_name = get_post_meta( $post_id, '_applicant_name', true );
		$applicant_email = get_post_meta( $post_id, '_applicant_email', true );
		$applicant_cv = get_post_meta( $post_id, '_applicant_cv', true );
		$job_id = get_post_meta( $post_id, '_job_applied_for', true );
		$job_title = get_the_title( $job_id );
		$application_data = get_post_meta( $post_id, '_humera_application_data', true );
		
        // Enqueue Print Assets for this detached view
        // Since we are not in standard admin flow, we have to make sure styles/scripts are printed
        
        // Enqueue Print Assets
		wp_enqueue_style( 'humera-print-styles', HUMERA_JOB_BOARD_URL . 'assets/css/print-styles.css', array(), '1.0.0' );
		wp_enqueue_script( 'humera-print-script', HUMERA_JOB_BOARD_URL . 'assets/js/print-script.js', array(), '1.0.0', true );
        
		?>
		<!DOCTYPE html>
		<html lang="en">
		<head>
			<meta charset="UTF-8">
			<meta name="viewport" content="width=device-width, initial-scale=1.0">
			<title>
				<?php 
				/* translators: %s: Applicant Name */
				printf( esc_html__( 'Application: %s', 'humera-job-board' ), esc_html( $applicant_name ) ); 
				?>
			</title>
            
            <?php 
            // Print Enqueued Styles
			wp_print_styles();
            ?>
		</head>
		<body>
			<button class="no-print" onclick="window.print()" style="padding: 10px 20px; font-size: 16px; margin-bottom: 20px; cursor: pointer;">🖨️ Print / Save as PDF</button>
			
			<h1><?php esc_html_e( 'Job Application', 'humera-job-board' ); ?></h1>
			
			<div class="meta-info">
				<div class="meta-item"><span class="label"><?php esc_html_e( 'Applicant Name:', 'humera-job-board' ); ?></span> <?php echo esc_html( $applicant_name ); ?></div>
				<div class="meta-item"><span class="label"><?php esc_html_e( 'Applicant Email:', 'humera-job-board' ); ?></span> <?php echo esc_html( $applicant_email ); ?></div>
				<div class="meta-item"><span class="label"><?php esc_html_e( 'Job Applied For:', 'humera-job-board' ); ?></span> <?php echo esc_html( $job_title ); ?></div>
				<div class="meta-item"><span class="label"><?php esc_html_e( 'Date:', 'humera-job-board' ); ?></span> <?php echo esc_html( get_the_date( '', $post ) ); ?></div>
			</div>
			
			<h3><?php esc_html_e( 'Application Details', 'humera-job-board' ); ?></h3>
			<div class="content">
				<?php 
				if ( ! empty( $application_data ) && is_array( $application_data ) ) {
					foreach ( $application_data as $key => $value ) {
						// Skip standard fields we already showed
						if ( in_array( $key, array( 'applicant_name', 'applicant_email' ) ) ) continue;
						
						$label = ucfirst( str_replace( '_', ' ', $key ) );
						echo '<p><strong>' . esc_html( $label ) . ':</strong> ' . esc_html( $value ) . '</p>';
					}
				} else {
					echo esc_html( $post->post_content );
				}
				?>
			</div>
			
			<?php if ( $applicant_cv ) : ?>
			<div class="meta-info">
				<strong><?php esc_html_e( 'Attached CV/Resume:', 'humera-job-board' ); ?></strong><br>
				<a href="<?php echo esc_url( $applicant_cv ); ?>"><?php echo esc_html( $applicant_cv ); ?></a>
			</div>
			<?php endif; ?>
			
			<div class="footer">
				<p>Generated by Humera Job Board Plugin</p>
			</div>
            
            <?php
            // Print Enqueued Scripts
			wp_print_scripts();
            ?>
		</body>
		</html>
		<?php
		exit;
	}

	public function add_application_counts_to_menu() {
		global $submenu;

		if ( isset( $submenu['edit.php?post_type=job_listing'] ) ) {
			// Count 'job_application` posts
			$count = wp_count_posts( 'job_application' );
			$total = $count->publish; 
			
			if ( $total > 0 ) {
				foreach ( $submenu['edit.php?post_type=job_listing'] as $key => $menu_item ) {
					if ( isset( $menu_item[2] ) && 'edit.php?post_type=job_application' === $menu_item[2] ) {
						$submenu['edit.php?post_type=job_listing'][ $key ][0] .= ' <span class="awaiting-mod count-' . esc_attr( $total ) . '"><span class="pending-count" aria-hidden="true">' . esc_html( $total ) . '</span></span>';
						break;
					}
				}
			}
		}
	}

	public function customize_post_row_actions( $actions, $post ) {
		if ( $post->post_type === 'job_application' ) {
			// Remove Edit and Quick Edit
			unset( $actions['edit'] );
			unset( $actions['inline hide-if-no-js'] ); // Quick Edit
			
			// Remove Duplicator (common slugs)
			unset( $actions['duplicate'] );
			unset( $actions['duplicator'] ); 
			
			$view_url = admin_url( 'admin-post.php?action=humera_print_application&post_id=' . $post->ID );
			
			// Add View and Download Options
			// Rebuilding the array to control order if needed, but appending is fine.
			
			// Prepend View?
			$new_actions = array();
			$new_actions['humera_view_application'] = '<a href="' . esc_url( $view_url ) . '" target="_blank">' . esc_html__( 'View Application', 'humera-job-board' ) . '</a>';
			$new_actions['humera_download_pdf'] = '<a href="' . esc_url( $view_url ) . '" target="_blank">' . esc_html__( 'Download PDF', 'humera-job-board' ) . '</a>';
			
			// Merge with remaining actions (like Trash)
			$actions = array_merge( $new_actions, $actions );
		}
		return $actions;
	}

	public function add_export_button( $which ) {
		global $typenow;
		if ( 'job_application' === $typenow && 'top' === $which ) {
			?>
			<div class="alignleft actions">
				<a href="<?php echo esc_url( admin_url( 'admin-post.php?action=humera_export_applications' ) ); ?>" class="button button-primary"><?php esc_html_e( 'Export to CSV', 'humera-job-board' ); ?></a>
			</div>
			<?php
		}
	}

	public function handle_export_applications() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Unauthorized access.', 'humera-job-board' ) );
		}

		// Get all applications
		$args = array(
			'post_type'      => 'job_application',
			'post_status'    => 'publish',
			'posts_per_page' => -1,
		);
		$query = new WP_Query( $args );

		// Set headers for file download
		header( 'Content-Type: text/csv; charset=utf-8' );
		header( 'Content-Disposition: attachment; filename=job-applications-' . date( 'Y-m-d' ) . '.csv' );
		header( 'Pragma: no-cache' );
		header( 'Expires: 0' );

		$output = fopen( 'php://output', 'w' );

		// CSV Headers
		fputcsv( $output, array( 'ID', 'Job Title', 'Applicant Name', 'Applicant Email', 'Date', 'CV URL' ) );

		while ( $query->have_posts() ) {
			$query->the_post(); // Sets global $post
			
			$post_id = get_the_ID();
			$job_id = get_post_meta( $post_id, '_job_applied_for', true );
			$job_title = get_the_title( $job_id );
			$name = get_post_meta( $post_id, '_applicant_name', true );
			$email = get_post_meta( $post_id, '_applicant_email', true );
			$cv = get_post_meta( $post_id, '_applicant_cv', true );

			fputcsv( $output, array( 
				$post_id, 
				$job_title, 
				$name, 
				$email, 
				get_the_date( 'Y-m-d H:i:s' ), 
				$cv 
			) );
		}

		fclose( $output );
		exit;
	}

}
