<?php

class Humera_Job_Dashboard {

	private $plugin_name;
	private $version;

	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version = $version;
	}

	public function render_dashboard_shortcode( $atts ) {
		if ( ! is_user_logged_in() ) {
			return '<p>' . esc_html__( 'You must be logged in to view this page.', 'humera-job-board' ) . '</p>';
		}

		if ( ! current_user_can( 'employer' ) && ! current_user_can( 'administrator' ) ) {
			return '<p>' . esc_html__( 'You do not have permission to view this page.', 'humera-job-board' ) . '</p>';
		}

		$action = isset( $_GET['job_action'] ) ? sanitize_text_field( wp_unslash( $_GET['job_action'] ) ) : 'list';
		$job_id = isset( $_GET['job_id'] ) ? intval( $_GET['job_id'] ) : 0;

		ob_start();

		if ( 'edit' === $action || 'add' === $action ) {
			$this->render_job_form( $job_id );
		} else {
			$this->render_job_list();
		}

		return ob_get_clean();
	}

	private function render_job_list() {
		$args = array(
			'post_type'      => 'job_listing',
			'post_status'    => array( 'publish', 'pending', 'draft' ),
			'author'         => get_current_user_id(),
			'posts_per_page' => -1,
		);

		$query = new WP_Query( $args );

		echo '<div class="humera-dashboard-header">';
		echo '<h2>' . esc_html__( 'My Jobs', 'humera-job-board' ) . '</h2>';
		echo '<a href="' . esc_url( add_query_arg( 'job_action', 'add' ) ) . '" class="humera-btn">' . esc_html__( 'Post a Job', 'humera-job-board' ) . '</a>';
		echo '</div>';

		if ( $query->have_posts() ) {
			echo '<table class="humera-dashboard-table">';
			echo '<thead><tr><th>' . esc_html__( 'Title', 'humera-job-board' ) . '</th><th>' . esc_html__( 'Status', 'humera-job-board' ) . '</th><th>' . esc_html__( 'Views', 'humera-job-board' ) . '</th><th>' . esc_html__( 'Applications', 'humera-job-board' ) . '</th><th>' . esc_html__( 'Date', 'humera-job-board' ) . '</th><th>' . esc_html__( 'Actions', 'humera-job-board' ) . '</th></tr></thead>';
			echo '<tbody>';
			while ( $query->have_posts() ) {
				$query->the_post();
				$job_id = get_the_ID();
				$edit_url = add_query_arg( array( 'job_action' => 'edit', 'job_id' => $job_id ) );
				$delete_url = wp_nonce_url( add_query_arg( array( 'action' => 'humera_delete_job', 'job_id' => $job_id ), admin_url( 'admin-post.php' ) ), 'humera_delete_job_' . $job_id );
				
				$views = get_post_meta( $job_id, '_humera_job_views', true );
				$views = $views ? $views : 0;

				$app_query = new WP_Query( array(
					'post_type' => 'job_application',
					'meta_key' => '_job_applied_for',
					'meta_value' => $job_id,
					'fields' => 'ids',
				) );
				$app_count = $app_query->found_posts;

				echo '<tr>';
				echo '<td>' . esc_html( get_the_title() ) . '</td>';
				echo '<td>' . esc_html( get_post_status_object( get_post_status() )->label ) . '</td>';
				echo '<td>' . esc_html( $views ) . '</td>';
				echo '<td>' . esc_html( $app_count ) . '</td>';
				echo '<td>' . esc_html( get_the_date() ) . '</td>';
				echo '<td>';
				echo '<a href="' . esc_url( $edit_url ) . '">' . esc_html__( 'Edit', 'humera-job-board' ) . '</a> | ';
				echo '<a href="' . esc_url( $delete_url ) . '" onclick="return confirm(\'' . esc_attr__( 'Are you sure?', 'humera-job-board' ) . '\');" class="humera-delete-link">' . esc_html__( 'Delete', 'humera-job-board' ) . '</a>';
				echo '</td>';
				echo '</tr>';
			}
			echo '</tbody>';
			echo '</table>';
			wp_reset_postdata();
		} else {
			echo '<p>' . esc_html__( 'You have not posted any jobs yet.', 'humera-job-board' ) . '</p>';
		}
	}

	private function render_job_form( $job_id = 0 ) {
		$title = '';
		$content = '';
		$location = '';
		$salary = '';
		$deadline = '';
		$email = '';
		$job_cat = array();
		$job_type = array();

		if ( $job_id ) {
			$post = get_post( $job_id );
			if ( $post && $post->post_author == get_current_user_id() ) {
				$title = $post->post_title;
				$content = $post->post_content;
				$location = get_post_meta( $job_id, '_humera_job_location', true );
				$salary = get_post_meta( $job_id, '_humera_job_salary', true );
				$deadline = get_post_meta( $job_id, '_humera_job_deadline', true );
				$email = get_post_meta( $job_id, '_humera_job_application_email', true );
				$job_cat = wp_get_object_terms( $job_id, 'job_category', array( 'fields' => 'ids' ) );
				$job_type = wp_get_object_terms( $job_id, 'job_type', array( 'fields' => 'ids' ) );
			}
		}

		echo '<h3>' . ( $job_id ? esc_html__( 'Edit Job', 'humera-job-board' ) : esc_html__( 'Post a Job', 'humera-job-board' ) ) . '</h3>';
		?>
		<form method="post" action="<?php echo esc_url( admin_url( 'admin-post.php' ) ); ?>" class="humera-job-form">
			<input type="hidden" name="action" value="humera_save_job">
			<input type="hidden" name="job_id" value="<?php echo esc_attr( $job_id ); ?>">
			<?php wp_nonce_field( 'humera_save_job', 'humera_job_nonce' ); ?>

			<p>
				<label><?php esc_html_e( 'Job Title', 'humera-job-board' ); ?></label>
				<input type="text" name="job_title" value="<?php echo esc_attr( $title ); ?>" required>
			</p>

			<p>
				<label><?php esc_html_e( 'Description', 'humera-job-board' ); ?></label>
				<?php wp_editor( $content, 'job_description', array( 'textarea_name' => 'job_description', 'media_buttons' => false, 'textarea_rows' => 10 ) ); ?>
			</p>

			<p>
				<label><?php esc_html_e( 'Location', 'humera-job-board' ); ?></label>
				<input type="text" name="job_location" value="<?php echo esc_attr( $location ); ?>">
			</p>

			<p>
				<label><?php esc_html_e( 'Salary', 'humera-job-board' ); ?></label>
				<input type="text" name="job_salary" value="<?php echo esc_attr( $salary ); ?>">
			</p>

			<p>
				<label><?php esc_html_e( 'Application Deadline', 'humera-job-board' ); ?></label>
				<input type="date" name="job_deadline" value="<?php echo esc_attr( $deadline ); ?>">
			</p>

			<p>
				<label><?php esc_html_e( 'Application Email', 'humera-job-board' ); ?></label>
				<input type="email" name="job_application_email" value="<?php echo esc_attr( $email ); ?>">
			</p>
			
			<!-- Categories and Types would go here, simplified for brevity -->

			<p>
				<button type="submit" class="humera-btn"><?php esc_html_e( 'Save Job', 'humera-job-board' ); ?></button>
				<a href="<?php echo esc_url( remove_query_arg( array( 'job_action', 'job_id' ) ) ); ?>" class="humera-cancel-link"><?php esc_html_e( 'Cancel', 'humera-job-board' ); ?></a>
			</p>
		</form>
		<?php
	}

	public function handle_save_job() {
		if ( ! isset( $_POST['humera_job_nonce'] ) || ! wp_verify_nonce( $_POST['humera_job_nonce'], 'humera_save_job' ) ) {
			wp_die( esc_html__( 'Security check failed', 'humera-job-board' ) );
		}

		if ( ! is_user_logged_in() ) {
			wp_die( esc_html__( 'You must be logged in', 'humera-job-board' ) );
		}

		$job_id = isset( $_POST['job_id'] ) ? intval( $_POST['job_id'] ) : 0;
		$title = isset( $_POST['job_title'] ) ? sanitize_text_field( wp_unslash( $_POST['job_title'] ) ) : '';
		$content = isset( $_POST['job_description'] ) ? wp_kses_post( wp_unslash( $_POST['job_description'] ) ) : '';

		$post_data = array(
			'post_title'   => $title,
			'post_content' => $content,
			'post_type'    => 'job_listing',
			'post_status'  => 'publish', // Or pending based on settings
		);

		if ( $job_id ) {
			$post = get_post( $job_id );
			if ( $post->post_author != get_current_user_id() && ! current_user_can( 'administrator' ) ) {
				wp_die( esc_html__( 'Unauthorized', 'humera-job-board' ) );
			}
			$post_data['ID'] = $job_id;
			wp_update_post( $post_data );
		} else {
			$job_id = wp_insert_post( $post_data );
		}

		if ( $job_id ) {
			if ( isset( $_POST['job_location'] ) ) {
				update_post_meta( $job_id, '_humera_job_location', sanitize_text_field( wp_unslash( $_POST['job_location'] ) ) );
			}
			if ( isset( $_POST['job_salary'] ) ) {
				update_post_meta( $job_id, '_humera_job_salary', sanitize_text_field( wp_unslash( $_POST['job_salary'] ) ) );
			}
			if ( isset( $_POST['job_deadline'] ) ) {
				update_post_meta( $job_id, '_humera_job_deadline', sanitize_text_field( wp_unslash( $_POST['job_deadline'] ) ) );
			}
			if ( isset( $_POST['job_application_email'] ) ) {
				update_post_meta( $job_id, '_humera_job_application_email', sanitize_email( wp_unslash( $_POST['job_application_email'] ) ) );
			}
		}

		wp_safe_redirect( remove_query_arg( array( 'job_action', 'job_id' ), wp_get_referer() ) );
		exit;
	}
	
	public function handle_delete_job() {
		$job_id = isset( $_GET['job_id'] ) ? intval( $_GET['job_id'] ) : 0;
		check_admin_referer( 'humera_delete_job_' . $job_id );

		if ( ! is_user_logged_in() ) {
			wp_die( esc_html__( 'You must be logged in', 'humera-job-board' ) );
		}

		$post = get_post( $job_id );
		if ( $post && ( $post->post_author == get_current_user_id() || current_user_can( 'administrator' ) ) ) {
			wp_trash_post( $job_id );
		}

		wp_safe_redirect( remove_query_arg( array( 'action', 'job_id', '_wpnonce' ), wp_get_referer() ) );
		exit;
	}

}
