<?php

class Humera_Job_Admin {

	private $plugin_name;
	private $version;

	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version = $version;
	}

	public function enqueue_styles() {
		wp_enqueue_style( $this->plugin_name, HUMERA_JOB_BOARD_URL . 'assets/css/humera-job-admin.css', array(), $this->version . '.2', 'all' );

		// Custom menu icon via CSS background-image (avoids WordPress opacity/filter bugs with <img> tags)
		$icon_url = HUMERA_JOB_BOARD_URL . 'assets/images/humera-job-board-icon.png';
		$custom_css = '
			#adminmenu .menu-icon-humera_job_listing .wp-menu-image::before {
				content: "" !important;
			}
			#adminmenu .menu-icon-humera_job_listing .wp-menu-image {
				background-image: url("' . esc_url( $icon_url ) . '") !important;
				background-size: 20px 20px !important;
				background-repeat: no-repeat !important;
				background-position: center center !important;
			}
		';
		wp_add_inline_style( $this->plugin_name, $custom_css );
	}

	public function enqueue_scripts() {
		wp_enqueue_script( $this->plugin_name, HUMERA_JOB_BOARD_URL . 'assets/js/humera-job-admin.js', array( 'jquery' ), $this->version, false );

            // Enqueue Form Editor Script only on the specific page
            // phpcs:ignore WordPress.Security.NonceVerification.Recommended -- Page check only
            if ( isset( $_GET['page'] ) && 'humera-job-form-editor' === $_GET['page'] ) {
            wp_enqueue_script( 'humera-job-form-editor', HUMERA_JOB_BOARD_URL . 'assets/js/admin-form-editor.js', array( 'jquery' ), $this->version, true );
            
            // Localize script to pass PHP data
            $saved_fields = get_option( 'humera_job_form_settings', array() );
            // Default fields count if empty
            if ( empty( $saved_fields ) ) {
                $count = 4; // Default validation count based on default array size in display_form_editor_page
            } else {
                $count = count( $saved_fields );
            }
            
            wp_localize_script( 'humera-job-form-editor', 'humeraJobAdmin', array(
                'fieldIndex' => $count
            ) );
        }
	}

	public function display_dashboard_page() {
		$active_jobs   = wp_count_posts( 'humera_job_listing' );
		$applications  = wp_count_posts( 'humera_app' );
		$categories    = wp_count_terms( array( 'taxonomy' => 'humera_job_category' ) );
		$job_types     = wp_count_terms( array( 'taxonomy' => 'humera_job_type' ) );
		$total_views   = 0;

		// Count total views across all jobs.
		$all_jobs_q = new WP_Query( array( 'post_type' => 'humera_job_listing', 'posts_per_page' => -1, 'fields' => 'ids' ) );
		if ( ! empty( $all_jobs_q->posts ) ) {
			foreach ( $all_jobs_q->posts as $jid ) {
				$total_views += intval( get_post_meta( $jid, '_humera_job_views', true ) );
			}
		}

		// Recent jobs.
		$recent_jobs = new WP_Query( array(
			'post_type'      => 'humera_job_listing',
			'posts_per_page' => 5,
			'orderby'        => 'date',
			'order'          => 'DESC',
		) );

		$current_user = wp_get_current_user();
		?>
		<div class="wrap humera-admin-dashboard">

			<!-- Welcome Banner -->
			<div class="humera-dash-welcome">
				<div class="humera-dash-welcome-text">
					<?php /* translators: %s: current user display name */ ?>
					<h1><?php printf( esc_html__( 'Welcome back, %s 👋', 'humera-job-board' ), esc_html( $current_user->display_name ) ); ?></h1>
					<p><?php esc_html_e( 'Here\'s what\'s happening with your job board today.', 'humera-job-board' ); ?></p>
				</div>
				<div class="humera-dash-welcome-actions">
					<a href="<?php echo esc_url( admin_url( 'post-new.php?post_type=humera_job_listing' ) ); ?>" class="button button-primary button-hero">
						<span class="dashicons dashicons-plus-alt2"></span>
						<?php esc_html_e( 'Add New Job', 'humera-job-board' ); ?>
					</a>
				</div>
			</div>

			<!-- Stat Cards -->
			<div class="humera-dashboard-stats">
				<div class="humera-stat-card humera-stat--active">
					<div class="stat-icon"><span class="dashicons dashicons-megaphone"></span></div>
					<div class="stat-content">
						<div class="stat-number"><?php echo intval( $active_jobs->publish ); ?></div>
						<div class="stat-label"><?php esc_html_e( 'Active Jobs', 'humera-job-board' ); ?></div>
					</div>
				</div>
				<div class="humera-stat-card humera-stat--draft">
					<div class="stat-icon"><span class="dashicons dashicons-edit"></span></div>
					<div class="stat-content">
						<div class="stat-number"><?php echo intval( $active_jobs->draft ); ?></div>
						<div class="stat-label"><?php esc_html_e( 'Drafts', 'humera-job-board' ); ?></div>
					</div>
				</div>
				<div class="humera-stat-card humera-stat--apps">
					<div class="stat-icon"><span class="dashicons dashicons-groups"></span></div>
					<div class="stat-content">
						<div class="stat-number"><?php echo intval( $applications->publish ); ?></div>
						<div class="stat-label"><?php esc_html_e( 'Applications', 'humera-job-board' ); ?></div>
					</div>
				</div>
				<div class="humera-stat-card humera-stat--views">
					<div class="stat-icon"><span class="dashicons dashicons-visibility"></span></div>
					<div class="stat-content">
						<div class="stat-number"><?php echo intval( $total_views ); ?></div>
						<div class="stat-label"><?php esc_html_e( 'Total Views', 'humera-job-board' ); ?></div>
					</div>
				</div>
				<div class="humera-stat-card humera-stat--cats">
					<div class="stat-icon"><span class="dashicons dashicons-category"></span></div>
					<div class="stat-content">
						<div class="stat-number"><?php echo intval( $categories ); ?></div>
						<div class="stat-label"><?php esc_html_e( 'Categories', 'humera-job-board' ); ?></div>
					</div>
				</div>
				<div class="humera-stat-card humera-stat--types">
					<div class="stat-icon"><span class="dashicons dashicons-tag"></span></div>
					<div class="stat-content">
						<div class="stat-number"><?php echo intval( $job_types ); ?></div>
						<div class="stat-label"><?php esc_html_e( 'Job Types', 'humera-job-board' ); ?></div>
					</div>
				</div>
			</div>

			<!-- Quick Actions -->
			<div class="humera-dashboard-actions">
				<h2><?php esc_html_e( 'Quick Actions', 'humera-job-board' ); ?></h2>
				<div class="humera-actions-grid">
					<a href="<?php echo esc_url( admin_url( 'edit.php?post_type=humera_job_listing' ) ); ?>" class="humera-action-card">
						<span class="dashicons dashicons-list-view"></span>
						<span><?php esc_html_e( 'All Jobs', 'humera-job-board' ); ?></span>
					</a>
					<a href="<?php echo esc_url( admin_url( 'edit.php?post_type=humera_app' ) ); ?>" class="humera-action-card">
						<span class="dashicons dashicons-email"></span>
						<span><?php esc_html_e( 'Applications', 'humera-job-board' ); ?></span>
					</a>
					<a href="<?php echo esc_url( admin_url( 'edit-tags.php?taxonomy=humera_job_category&post_type=humera_job_listing' ) ); ?>" class="humera-action-card">
						<span class="dashicons dashicons-category"></span>
						<span><?php esc_html_e( 'Categories', 'humera-job-board' ); ?></span>
					</a>
					<a href="<?php echo esc_url( admin_url( 'edit-tags.php?taxonomy=humera_job_type&post_type=humera_job_listing' ) ); ?>" class="humera-action-card">
						<span class="dashicons dashicons-tag"></span>
						<span><?php esc_html_e( 'Job Types', 'humera-job-board' ); ?></span>
					</a>
				</div>
			</div>

			<!-- Recent Jobs -->
			<div class="humera-dashboard-recent">
				<div class="humera-recent-header">
					<h2><?php esc_html_e( 'Recent Jobs', 'humera-job-board' ); ?></h2>
					<a href="<?php echo esc_url( admin_url( 'edit.php?post_type=humera_job_listing' ) ); ?>" class="humera-view-all-link">
						<?php esc_html_e( 'View All →', 'humera-job-board' ); ?>
					</a>
				</div>
				<?php if ( $recent_jobs->have_posts() ) : ?>
					<table class="humera-recent-table">
						<thead>
							<tr>
								<th><?php esc_html_e( 'Job Title', 'humera-job-board' ); ?></th>
								<th><?php esc_html_e( 'Location', 'humera-job-board' ); ?></th>
								<th><?php esc_html_e( 'Status', 'humera-job-board' ); ?></th>
								<th><?php esc_html_e( 'Views', 'humera-job-board' ); ?></th>
								<th><?php esc_html_e( 'Date', 'humera-job-board' ); ?></th>
							</tr>
						</thead>
						<tbody>
							<?php while ( $recent_jobs->have_posts() ) : $recent_jobs->the_post(); ?>
								<?php
									$job_location = get_post_meta( get_the_ID(), '_humera_job_location', true );
									$job_views    = get_post_meta( get_the_ID(), '_humera_job_views', true );
									$status       = get_post_status();
									$status_class = 'humera-status--' . $status;
								?>
								<tr>
									<td>
										<a href="<?php echo esc_url( get_edit_post_link() ); ?>" class="humera-job-title-link">
											<?php echo esc_html( get_the_title() ); ?>
										</a>
									</td>
									<td>
										<?php if ( $job_location ) : ?>
											<span class="humera-location-cell">
												<span class="dashicons dashicons-location"></span>
												<?php echo esc_html( $job_location ); ?>
											</span>
										<?php else : ?>
											<span class="humera-muted">—</span>
										<?php endif; ?>
									</td>
									<td>
										<span class="humera-status-pill <?php echo esc_attr( $status_class ); ?>">
											<?php echo esc_html( ucfirst( $status ) ); ?>
										</span>
									</td>
									<td>
										<span class="humera-views-cell">
											<span class="dashicons dashicons-visibility"></span>
											<?php echo intval( $job_views ); ?>
										</span>
									</td>
									<td><?php echo esc_html( get_the_date( 'M j, Y' ) ); ?></td>
								</tr>
							<?php endwhile; wp_reset_postdata(); ?>
						</tbody>
					</table>
				<?php else : ?>
					<div class="humera-empty-state">
						<span class="dashicons dashicons-clipboard"></span>
						<p><?php esc_html_e( 'No jobs posted yet. Create your first job listing!', 'humera-job-board' ); ?></p>
						<a href="<?php echo esc_url( admin_url( 'post-new.php?post_type=humera_job_listing' ) ); ?>" class="button button-primary">
							<?php esc_html_e( 'Add New Job', 'humera-job-board' ); ?>
						</a>
					</div>
				<?php endif; ?>
			</div>

		</div>
		<?php
	}

	public function display_plugin_setup_page() {
		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'Humera Job Board - Help & Shortcodes', 'humera-job-board' ); ?></h1>
			
			<div class="card">
				<h2><?php esc_html_e( 'Available Shortcodes', 'humera-job-board' ); ?></h2>
				
				<table class="form-table" role="presentation">
					<tbody>
						<tr>
							<th scope="row"><code>[humera_jobs]</code></th>
							<td>
								<p><?php esc_html_e( 'Displays the list of latest job postings with a search form.', 'humera-job-board' ); ?></p>
								<p><strong><?php esc_html_e( 'Attributes:', 'humera-job-board' ); ?></strong></p>
								<ul>
									<li><code>posts_per_page="10"</code> - <?php esc_html_e( 'Number of jobs to show per page (default: 10).', 'humera-job-board' ); ?></li>
								</ul>
								<p><em><?php esc_html_e( 'Example:', 'humera-job-board' ); ?></em> <code>[humera_jobs posts_per_page="5"]</code></p>
							</td>
						</tr>
						<tr>
							<th scope="row"><code>[humera_employer_dashboard]</code></th>
							<td>
								<p><?php esc_html_e( 'Displays the frontend dashboard for employers to manage their posted jobs.', 'humera-job-board' ); ?></p>
							</td>
						</tr>
					</tbody>
				</table>
			</div>

			<div class="card">
				<h2><?php esc_html_e( 'Customization Guide', 'humera-job-board' ); ?></h2>
				<p><?php echo wp_kses_post( __( 'You can customize the application form fields using the <code>humera_job_application_fields</code> filter in your theme functions.', 'humera-job-board' ) ); ?></p>
			</div>
		</div>
		<?php
	}

	public function add_plugin_admin_menu() {
		// Dashboard as first submenu (replaces the default 'All Jobs' link when clicking 'Job Listings')
		add_submenu_page(
			'edit.php?post_type=humera_job_listing',
			__( 'Dashboard', 'humera-job-board' ),
			__( 'Dashboard', 'humera-job-board' ),
			'manage_options',
			'humera-job-dashboard',
			array( $this, 'display_dashboard_page' )
		);

		add_submenu_page(
			'edit.php?post_type=humera_job_listing',
			__( 'Help & Shortcodes', 'humera-job-board' ),
			__( 'Help & Shortcodes', 'humera-job-board' ),
			'manage_options',
			'humera-job-help',
			array( $this, 'display_plugin_setup_page' )
		);
		
		add_submenu_page(
			'edit.php?post_type=humera_job_listing',
			__( 'Form Editor', 'humera-job-board' ),
			__( 'Form Editor', 'humera-job-board' ),
			'manage_options',
			'humera-job-form-editor',
			array( $this, 'display_form_editor_page' )
		);

		add_submenu_page(
			'edit.php?post_type=humera_job_listing',
			__( 'SMTP Settings', 'humera-job-board' ),
			__( 'SMTP Settings', 'humera-job-board' ),
			'manage_options',
			'humera-job-smtp',
			array( $this, 'display_smtp_settings_page' )
		);
	}

	/**
	 * Redirect the top-level "Job Listings" click to the Dashboard page.
	 */
	public function redirect_to_dashboard() {
		global $pagenow;

		if ( 'edit.php' !== $pagenow ) {
			return;
		}

		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$post_type = isset( $_GET['post_type'] ) ? sanitize_text_field( wp_unslash( $_GET['post_type'] ) ) : '';
		// phpcs:ignore WordPress.Security.NonceVerification.Recommended
		$page = isset( $_GET['page'] ) ? sanitize_text_field( wp_unslash( $_GET['page'] ) ) : '';

		// Only redirect from the main "All Jobs" listing — not from submenu pages or other actions.
		// phpcs:disable WordPress.Security.NonceVerification.Recommended -- Read-only page-routing check, no data processing.
		if (
			'humera_job_listing' === $post_type
			&& empty( $page )
			&& ! isset( $_GET['s'] )
			&& ! isset( $_GET['action'] )
			&& ! isset( $_GET['filter_action'] )
			&& ! isset( $_GET['trashed'] )
			&& ! isset( $_GET['untrashed'] )
			&& ! isset( $_GET['deleted'] )
		) {
			// phpcs:enable WordPress.Security.NonceVerification.Recommended
			wp_safe_redirect( admin_url( 'edit.php?post_type=humera_job_listing&page=humera-job-dashboard' ) );
			exit;
		}
	}

	/**
	 * Display the SMTP Settings admin page.
	 */
	public function display_smtp_settings_page() {
		// Handle Save
		if ( isset( $_POST['humera_save_smtp'] ) && check_admin_referer( 'humera_smtp_nonce', 'humera_smtp_nonce_field' ) ) {
			$settings = array(
				'enabled'    => ! empty( $_POST['humera_smtp_enabled'] ) ? 1 : 0,
				'host'       => isset( $_POST['humera_smtp_host'] ) ? sanitize_text_field( wp_unslash( $_POST['humera_smtp_host'] ) ) : '',
				'port'       => isset( $_POST['humera_smtp_port'] ) ? absint( $_POST['humera_smtp_port'] ) : 587,
				'encryption' => isset( $_POST['humera_smtp_encryption'] ) ? sanitize_text_field( wp_unslash( $_POST['humera_smtp_encryption'] ) ) : 'tls',
				'auth'       => ! empty( $_POST['humera_smtp_auth'] ) ? 1 : 0,
				'username'   => isset( $_POST['humera_smtp_username'] ) ? sanitize_text_field( wp_unslash( $_POST['humera_smtp_username'] ) ) : '',
				'from_email' => isset( $_POST['humera_smtp_from_email'] ) ? sanitize_email( wp_unslash( $_POST['humera_smtp_from_email'] ) ) : '',
				'from_name'  => isset( $_POST['humera_smtp_from_name'] ) ? sanitize_text_field( wp_unslash( $_POST['humera_smtp_from_name'] ) ) : '',
			);

			// Only update password if it was changed (not the placeholder).
			$password_input = isset( $_POST['humera_smtp_password'] ) ? sanitize_text_field( wp_unslash( $_POST['humera_smtp_password'] ) ) : ''; // phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
			if ( '' !== $password_input && '********' !== $password_input ) {
				$settings['password'] = base64_encode( $password_input ); // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_encode -- Simple obfuscation, not security.
			} else {
				$existing = get_option( 'humera_smtp_settings', array() );
				$settings['password'] = isset( $existing['password'] ) ? $existing['password'] : '';
			}

			update_option( 'humera_smtp_settings', $settings );
			echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'SMTP settings saved.', 'humera-job-board' ) . '</p></div>';
		}

		// Handle Test Email
		if ( isset( $_POST['humera_send_test_email'] ) && check_admin_referer( 'humera_smtp_nonce', 'humera_smtp_nonce_field' ) ) {
			$test_to = isset( $_POST['humera_test_email'] ) ? sanitize_email( wp_unslash( $_POST['humera_test_email'] ) ) : '';
			if ( $test_to ) {
				$result = wp_mail(
					$test_to,
					__( 'Humera Job Board — SMTP Test Email', 'humera-job-board' ),
					__( "This is a test email from Humera Job Board.\n\nIf you received this, your SMTP settings are working correctly!", 'humera-job-board' )
				);
				if ( $result ) {
					echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Test email sent successfully! Check your inbox.', 'humera-job-board' ) . '</p></div>';
				} else {
					echo '<div class="notice notice-error is-dismissible"><p>' . esc_html__( 'Failed to send test email. Please check your SMTP settings.', 'humera-job-board' ) . '</p></div>';
				}
			}
		}

		$s = get_option( 'humera_smtp_settings', array() );
		$enabled    = isset( $s['enabled'] ) ? $s['enabled'] : 0;
		$host       = isset( $s['host'] ) ? $s['host'] : '';
		$port       = isset( $s['port'] ) ? $s['port'] : 587;
		$encryption = isset( $s['encryption'] ) ? $s['encryption'] : 'tls';
		$auth       = isset( $s['auth'] ) ? $s['auth'] : 1;
		$username   = isset( $s['username'] ) ? $s['username'] : '';
		$password   = isset( $s['password'] ) && $s['password'] ? '********' : '';
		$from_email = isset( $s['from_email'] ) ? $s['from_email'] : '';
		$from_name  = isset( $s['from_name'] ) ? $s['from_name'] : '';
		?>
		<div class="wrap humera-admin-dashboard">
			<h1><?php esc_html_e( 'SMTP Email Settings', 'humera-job-board' ); ?></h1>
			<p><?php esc_html_e( 'Configure SMTP to ensure application notification emails are delivered reliably.', 'humera-job-board' ); ?></p>

			<form method="post" action="">
				<?php wp_nonce_field( 'humera_smtp_nonce', 'humera_smtp_nonce_field' ); ?>

				<table class="form-table">
					<tr>
						<th scope="row"><?php esc_html_e( 'Enable SMTP', 'humera-job-board' ); ?></th>
						<td>
							<label>
								<input type="checkbox" name="humera_smtp_enabled" value="1" <?php checked( $enabled, 1 ); ?> />
								<?php esc_html_e( 'Use SMTP for sending emails from this plugin', 'humera-job-board' ); ?>
							</label>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="humera_smtp_host"><?php esc_html_e( 'SMTP Host', 'humera-job-board' ); ?></label></th>
						<td><input type="text" name="humera_smtp_host" id="humera_smtp_host" value="<?php echo esc_attr( $host ); ?>" class="regular-text" placeholder="smtp.gmail.com" /></td>
					</tr>
					<tr>
						<th scope="row"><label for="humera_smtp_port"><?php esc_html_e( 'SMTP Port', 'humera-job-board' ); ?></label></th>
						<td><input type="number" name="humera_smtp_port" id="humera_smtp_port" value="<?php echo esc_attr( $port ); ?>" class="small-text" placeholder="587" /></td>
					</tr>
					<tr>
						<th scope="row"><?php esc_html_e( 'Encryption', 'humera-job-board' ); ?></th>
						<td>
							<label><input type="radio" name="humera_smtp_encryption" value="none" <?php checked( $encryption, 'none' ); ?> /> <?php esc_html_e( 'None', 'humera-job-board' ); ?></label>&nbsp;&nbsp;
							<label><input type="radio" name="humera_smtp_encryption" value="ssl" <?php checked( $encryption, 'ssl' ); ?> /> <?php esc_html_e( 'SSL', 'humera-job-board' ); ?></label>&nbsp;&nbsp;
							<label><input type="radio" name="humera_smtp_encryption" value="tls" <?php checked( $encryption, 'tls' ); ?> /> <?php esc_html_e( 'TLS', 'humera-job-board' ); ?></label>
						</td>
					</tr>
					<tr>
						<th scope="row"><?php esc_html_e( 'Authentication', 'humera-job-board' ); ?></th>
						<td>
							<label>
								<input type="checkbox" name="humera_smtp_auth" value="1" <?php checked( $auth, 1 ); ?> />
								<?php esc_html_e( 'Requires authentication', 'humera-job-board' ); ?>
							</label>
						</td>
					</tr>
					<tr>
						<th scope="row"><label for="humera_smtp_username"><?php esc_html_e( 'SMTP Username', 'humera-job-board' ); ?></label></th>
						<td><input type="text" name="humera_smtp_username" id="humera_smtp_username" value="<?php echo esc_attr( $username ); ?>" class="regular-text" autocomplete="off" /></td>
					</tr>
					<tr>
						<th scope="row"><label for="humera_smtp_password"><?php esc_html_e( 'SMTP Password', 'humera-job-board' ); ?></label></th>
						<td><input type="password" name="humera_smtp_password" id="humera_smtp_password" value="<?php echo esc_attr( $password ); ?>" class="regular-text" autocomplete="new-password" /></td>
					</tr>
					<tr>
						<th scope="row"><label for="humera_smtp_from_email"><?php esc_html_e( 'From Email', 'humera-job-board' ); ?></label></th>
						<td><input type="email" name="humera_smtp_from_email" id="humera_smtp_from_email" value="<?php echo esc_attr( $from_email ); ?>" class="regular-text" placeholder="noreply@yourdomain.com" /></td>
					</tr>
					<tr>
						<th scope="row"><label for="humera_smtp_from_name"><?php esc_html_e( 'From Name', 'humera-job-board' ); ?></label></th>
						<td><input type="text" name="humera_smtp_from_name" id="humera_smtp_from_name" value="<?php echo esc_attr( $from_name ); ?>" class="regular-text" placeholder="Humera Job Board" /></td>
					</tr>
				</table>

				<?php submit_button( __( 'Save SMTP Settings', 'humera-job-board' ), 'primary', 'humera_save_smtp' ); ?>

				<hr />

				<h2><?php esc_html_e( 'Send Test Email', 'humera-job-board' ); ?></h2>
				<p><?php esc_html_e( 'Save your settings above first, then send a test email to verify everything works.', 'humera-job-board' ); ?></p>
				<table class="form-table">
					<tr>
						<th scope="row"><label for="humera_test_email"><?php esc_html_e( 'Send To', 'humera-job-board' ); ?></label></th>
						<td>
							<input type="email" name="humera_test_email" id="humera_test_email" value="<?php echo esc_attr( get_option( 'admin_email' ) ); ?>" class="regular-text" />
							<?php submit_button( __( 'Send Test Email', 'humera-job-board' ), 'secondary', 'humera_send_test_email', false ); ?>
						</td>
					</tr>
				</table>
			</form>
		</div>
		<?php
	}

	/**
	 * Configure PHPMailer to use SMTP if enabled in plugin settings.
	 *
	 * @param PHPMailer $phpmailer The PHPMailer instance.
	 */
	public function configure_smtp( $phpmailer ) {
		$s = get_option( 'humera_smtp_settings', array() );

		if ( empty( $s['enabled'] ) || empty( $s['host'] ) ) {
			return;
		}

		$phpmailer->isSMTP();
		$phpmailer->Host       = $s['host']; // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
		$phpmailer->Port       = ! empty( $s['port'] ) ? (int) $s['port'] : 587; // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
		$phpmailer->SMTPSecure = ( isset( $s['encryption'] ) && 'none' !== $s['encryption'] ) ? $s['encryption'] : ''; // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
		$phpmailer->SMTPAuth   = ! empty( $s['auth'] ); // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase

		if ( ! empty( $s['auth'] ) ) {
			$phpmailer->Username = ! empty( $s['username'] ) ? $s['username'] : ''; // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
			$phpmailer->Password = ! empty( $s['password'] ) ? base64_decode( $s['password'] ) : ''; // phpcs:ignore WordPress.PHP.DiscouragedPHPFunctions.obfuscation_base64_decode, WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
		}

		if ( ! empty( $s['from_email'] ) ) {
			$phpmailer->From     = $s['from_email']; // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
			$phpmailer->FromName = ! empty( $s['from_name'] ) ? $s['from_name'] : 'Humera Job Board'; // phpcs:ignore WordPress.NamingConventions.ValidVariableName.UsedPropertyNotSnakeCase
		}
	}

	public function display_form_editor_page() {
		// Handle Save
		if ( isset( $_POST['humera_save_form_fields'] ) && check_admin_referer( 'humera_save_form_fields_nonce', 'humera_save_form_fields' ) ) {
			if ( ! empty( $_POST['fields'] ) ) {
				// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized -- Array structure sanitized in loop
				$fields = isset( $_POST['fields'] ) ? wp_unslash( $_POST['fields'] ) : array();
				$sanitized_fields = array();
				
				if ( is_array( $fields ) ) {
					foreach ( $fields as $key => $field ) {
						// Use specific keys enabled in the UI
						if ( ! empty( $field['key'] ) ) {
							$field_key = sanitize_title( $field['key'] );
							$sanitized_fields[ $field_key ] = array(
								'label'    => sanitize_text_field( $field['label'] ),
								'type'     => sanitize_text_field( $field['type'] ),
								'required' => isset( $field['required'] ),
							);
						}
					}
					update_option( 'humera_job_form_settings', $sanitized_fields );
					echo '<div class="notice notice-success is-dismissible"><p>' . esc_html__( 'Form fields saved successfully.', 'humera-job-board' ) . '</p></div>';
				}
			}
		}

		// Get current fields or defaults
		$saved_fields = get_option( 'humera_job_form_settings', array() );
		
		// If empty, populate with defaults for display
		if ( empty( $saved_fields ) ) {
			$saved_fields = array(
				'applicant_name' => array( 'label' => 'Full Name', 'type' => 'text', 'required' => true ),
				'applicant_email' => array( 'label' => 'Email Address', 'type' => 'email', 'required' => true ),
				'applicant_message' => array( 'label' => 'Cover Letter / Message', 'type' => 'textarea', 'required' => false ),
				'applicant_cv' => array( 'label' => 'Upload CV', 'type' => 'file', 'required' => true ),
			);
		}
		?>
		<div class="wrap">
			<h1><?php esc_html_e( 'Application Form Editor', 'humera-job-board' ); ?></h1>
			<p><?php esc_html_e( 'Customize the fields that appear on the job application form.', 'humera-job-board' ); ?></p>
			
			<form method="post" action="">
				<?php wp_nonce_field( 'humera_save_form_fields_nonce', 'humera_save_form_fields' ); ?>
				
				<table class="widefat fixed striped" id="humera-fields-table">
					<thead>
						<tr>
							<th style="width: 20%;"><?php esc_html_e( 'Field Key (Unique)', 'humera-job-board' ); ?></th>
							<th style="width: 30%;"><?php esc_html_e( 'Label', 'humera-job-board' ); ?></th>
							<th style="width: 20%;"><?php esc_html_e( 'Type', 'humera-job-board' ); ?></th>
							<th style="width: 10%;"><?php esc_html_e( 'Required', 'humera-job-board' ); ?></th>
							<th style="width: 10%;"><?php esc_html_e( 'Actions', 'humera-job-board' ); ?></th>
						</tr>
					</thead>
					<tbody id="humera-fields-body">
						<?php 
						$index = 0;
						foreach ( $saved_fields as $key => $field ) : 
						?>
						<tr class="field-row">
							<td><input type="text" name="fields[<?php echo esc_attr( $index ); ?>][key]" value="<?php echo esc_attr( $key ); ?>" class="regular-text" required ></td>
							<td><input type="text" name="fields[<?php echo esc_attr( $index ); ?>][label]" value="<?php echo esc_attr( $field['label'] ); ?>" class="regular-text" required></td>
							<td>
								<select name="fields[<?php echo esc_attr( $index ); ?>][type]">
									<option value="text" <?php selected( $field['type'], 'text' ); ?>>Text</option>
									<option value="email" <?php selected( $field['type'], 'email' ); ?>>Email</option>
									<option value="textarea" <?php selected( $field['type'], 'textarea' ); ?>>Textarea</option>
									<option value="file" <?php selected( $field['type'], 'file' ); ?>>File Select</option>
								</select>
							</td>
							<td><input type="checkbox" name="fields[<?php echo esc_attr( $index ); ?>][required]" <?php checked( ! empty( $field['required'] ) ); ?>></td>
							<td><button type="button" class="button humera-remove-field">Remove</button></td>
						</tr>
						<?php 
						$index++;
						endforeach; 
						?>
					</tbody>
				</table>
				
				<p>
					<button type="button" class="button" id="humera-add-field"><?php esc_html_e( '+ Add Field', 'humera-job-board' ); ?></button>
				</p>
				
				<hr>
				<p>
					<button type="submit" class="button button-primary"><?php esc_html_e( 'Save Form Fields', 'humera-job-board' ); ?></button>
				</p>
			</form>
		</div>
		<?php
	}
	
	public function add_meta_boxes() {
		add_meta_box(
			'humera_app_actions',
			__( 'Actions', 'humera-job-board' ),
			array( $this, 'render_application_actions_meta_box' ),
			'humera_app',
			'side',
			'high'
		);
	}
	
	public function render_application_actions_meta_box( $post ) {
		$print_url = wp_nonce_url( admin_url( 'admin-post.php?action=humera_print_application&post_id=' . $post->ID ), 'humera_print_application_' . $post->ID );
		?>
		<div class="misc-pub-section">
			<a href="<?php echo esc_url( $print_url ); ?>" target="_blank" class="button button-primary button-large" style="width: 100%; text-align: center;">
				<?php esc_html_e( 'Download / Print Application', 'humera-job-board' ); ?>
			</a>
		</div>
		<?php
	}
	
	public function handle_print_application() {
		// Verify nonce for print security
		$post_id = isset( $_GET['post_id'] ) ? intval( $_GET['post_id'] ) : 0;
		check_admin_referer( 'humera_print_application_' . $post_id );

		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Unauthorized access.', 'humera-job-board' ) );
		}
		
		$post_id = isset( $_GET['post_id'] ) ? intval( $_GET['post_id'] ) : 0;
		if ( ! $post_id ) {
			wp_die( esc_html__( 'Invalid Post ID.', 'humera-job-board' ) );
		}
		
		$post = get_post( $post_id );
		if ( ! $post || $post->post_type !== 'humera_app' ) {
			wp_die( esc_html__( 'Application not found.', 'humera-job-board' ) );
		}
		
		$applicant_name = get_post_meta( $post_id, '_applicant_name', true );
		$applicant_email = get_post_meta( $post_id, '_applicant_email', true );
		$applicant_cv = get_post_meta( $post_id, '_applicant_cv', true );
		$job_id = get_post_meta( $post_id, '_job_applied_for', true );
		$job_title = get_the_title( $job_id );
		$application_data = get_post_meta( $post_id, '_humera_application_data', true );
		
        // Enqueue Print Assets for this detached view
        // Since we are not in standard admin flow, we have to make sure styles/scripts are printed
        
        // Enqueue Print Assets
		wp_enqueue_style( 'humera-print-styles', HUMERA_JOB_BOARD_URL . 'assets/css/print-styles.css', array(), '1.0.0' );
		wp_enqueue_script( 'humera-print-script', HUMERA_JOB_BOARD_URL . 'assets/js/print-script.js', array(), '1.0.0', true );
        
		?>
		<!DOCTYPE html>
		<html lang="en">
		<head>
			<meta charset="UTF-8">
			<meta name="viewport" content="width=device-width, initial-scale=1.0">
			<title>
				<?php 
				/* translators: %s: Applicant Name */
				printf( esc_html__( 'Application: %s', 'humera-job-board' ), esc_html( $applicant_name ) ); 
				?>
			</title>
            
            <?php 
            // Print Enqueued Styles
			wp_print_styles();
            ?>
		</head>
		<body>
			<button class="no-print" onclick="window.print()" style="padding: 10px 20px; font-size: 16px; margin-bottom: 20px; cursor: pointer;">🖨️ Print / Save as PDF</button>
			
			<h1><?php esc_html_e( 'Job Application', 'humera-job-board' ); ?></h1>
			
			<div class="meta-info">
				<div class="meta-item"><span class="label"><?php esc_html_e( 'Applicant Name:', 'humera-job-board' ); ?></span> <?php echo esc_html( $applicant_name ); ?></div>
				<div class="meta-item"><span class="label"><?php esc_html_e( 'Applicant Email:', 'humera-job-board' ); ?></span> <?php echo esc_html( $applicant_email ); ?></div>
				<div class="meta-item"><span class="label"><?php esc_html_e( 'Job Applied For:', 'humera-job-board' ); ?></span> <?php echo esc_html( $job_title ); ?></div>
				<div class="meta-item"><span class="label"><?php esc_html_e( 'Date:', 'humera-job-board' ); ?></span> <?php echo esc_html( get_the_date( '', $post ) ); ?></div>
			</div>
			
			<h3><?php esc_html_e( 'Application Details', 'humera-job-board' ); ?></h3>
			<div class="content">
				<?php 
				if ( ! empty( $application_data ) && is_array( $application_data ) ) {
					foreach ( $application_data as $key => $value ) {
						// Skip standard fields we already showed
						if ( in_array( $key, array( 'applicant_name', 'applicant_email' ) ) ) continue;
						
						$label = ucfirst( str_replace( '_', ' ', $key ) );
						echo '<p><strong>' . esc_html( $label ) . ':</strong> ' . esc_html( $value ) . '</p>';
					}
				} else {
					echo wp_kses_post( $post->post_content );
				}
				?>
			</div>
			
			<?php if ( $applicant_cv ) : ?>
			<div class="meta-info">
				<strong><?php esc_html_e( 'Attached CV/Resume:', 'humera-job-board' ); ?></strong><br>
				<a href="<?php echo esc_url( $applicant_cv ); ?>"><?php echo esc_html( $applicant_cv ); ?></a>
			</div>
			<?php endif; ?>
			
			<div class="footer">
				<p>Generated by Humera Job Board Plugin</p>
			</div>
            
            <?php
            // Print Enqueued Scripts
			wp_print_scripts();
            ?>
		</body>
		</html>
		<?php
		exit;
	}

	public function add_application_counts_to_menu() {
		global $submenu;

		if ( isset( $submenu['edit.php?post_type=humera_job_listing'] ) ) {
			// Count 'humera_app' posts
			$count = wp_count_posts( 'humera_app' );
			$total = $count->publish; 
			
			if ( $total > 0 ) {
				foreach ( $submenu['edit.php?post_type=humera_job_listing'] as $key => $menu_item ) {
					if ( isset( $menu_item[2] ) && 'edit.php?post_type=humera_app' === $menu_item[2] ) {
						$submenu['edit.php?post_type=humera_job_listing'][ $key ][0] .= ' <span class="awaiting-mod count-' . esc_attr( $total ) . '"><span class="pending-count" aria-hidden="true">' . esc_html( $total ) . '</span></span>';
						break;
					}
				}
			}
		}
	}

	public function customize_post_row_actions( $actions, $post ) {
		if ( $post->post_type === 'humera_app' ) {
			// Remove Edit and Quick Edit
			unset( $actions['edit'] );
			unset( $actions['inline hide-if-no-js'] ); // Quick Edit
			
			// Remove Duplicator (common slugs)
			unset( $actions['duplicate'] );
			unset( $actions['duplicator'] ); 
			
			$view_url = wp_nonce_url( admin_url( 'admin-post.php?action=humera_print_application&post_id=' . $post->ID ), 'humera_print_application_' . $post->ID );
			
			// Add View and Download Options
			// Rebuilding the array to control order if needed, but appending is fine.
			
			// Prepend View?
			$new_actions = array();
			$new_actions['humera_view_application'] = '<a href="' . esc_url( $view_url ) . '" target="_blank">' . esc_html__( 'View Application', 'humera-job-board' ) . '</a>';
			$new_actions['humera_download_pdf'] = '<a href="' . esc_url( $view_url ) . '" target="_blank">' . esc_html__( 'Download PDF', 'humera-job-board' ) . '</a>';
			
			// Merge with remaining actions (like Trash)
			$actions = array_merge( $new_actions, $actions );
		}
		return $actions;
	}

	public function add_export_button( $which ) {
		global $typenow;
		if ( 'humera_app' === $typenow && 'top' === $which ) {
			?>
			<div class="alignleft actions">
				<a href="<?php echo esc_url( admin_url( 'admin-post.php?action=humera_export_applications' ) ); ?>" class="button button-primary"><?php esc_html_e( 'Export to CSV', 'humera-job-board' ); ?></a>
			</div>
			<?php
		}
	}

	public function handle_export_applications() {
		if ( ! current_user_can( 'manage_options' ) ) {
			wp_die( esc_html__( 'Unauthorized access.', 'humera-job-board' ) );
		}

		// Get all applications
		$args = array(
			'post_type'      => 'humera_app',
			'post_status'    => 'publish',
			'posts_per_page' => -1,
		);
		$query = new WP_Query( $args );

		// Set headers for file download
		header( 'Content-Type: text/csv; charset=utf-8' );
		header( 'Content-Disposition: attachment; filename=job-applications-' . gmdate( 'Y-m-d' ) . '.csv' );
		header( 'Pragma: no-cache' );
		header( 'Expires: 0' );

		// Manual CSV generation to avoid filesystem warnings
		$data_rows = array();
		
		// Headers
		$data_rows[] = '"ID","Job Title","Applicant Name","Applicant Email","Date","CV URL"';

		while ( $query->have_posts() ) {
			$query->the_post(); // Sets global $post
			
			$post_id = get_the_ID();
			$job_id = get_post_meta( $post_id, '_job_applied_for', true );
			$job_title = get_the_title( $job_id );
			$name = get_post_meta( $post_id, '_applicant_name', true );
			$email = get_post_meta( $post_id, '_applicant_email', true );
			$cv = get_post_meta( $post_id, '_applicant_cv', true );

			// Escape double quotes by doubling them
			$row = array(
				$post_id, 
				str_replace( '"', '""', $job_title ), 
				str_replace( '"', '""', $name ), 
				str_replace( '"', '""', $email ), 
				get_the_date( 'Y-m-d H:i:s' ), 
				str_replace( '"', '""', $cv ) 
			);
			
			$data_rows[] = '"' . implode( '","', $row ) . '"';
		}
		
		echo implode( "\n", $data_rows ); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		exit;
	}

}
