<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Humera_Job_Public {

	private $plugin_name;
	private $version;

	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version = $version;
	}

	public function enqueue_styles() {
		wp_enqueue_style( $this->plugin_name, HUMERA_JOB_BOARD_URL . 'assets/css/humera-job-public.css', array(), $this->version, 'all' );
	}

	public function enqueue_scripts() {
		wp_enqueue_script( $this->plugin_name, HUMERA_JOB_BOARD_URL . 'assets/js/humera-job-public.js', array( 'jquery' ), $this->version, true );
	}

	public function render_jobs_shortcode( $atts ) {
		$atts = shortcode_atts( array(
			'posts_per_page' => 10,
			'layout'         => 'grid', // 'grid' or 'list'
		), $atts, 'humera_jobs' );

		$paged = ( get_query_var( 'paged' ) ) ? get_query_var( 'paged' ) : 1;

		// phpcs:disable WordPress.Security.NonceVerification.Recommended -- Public search/sort form
		$sort = isset( $_GET['hjb_sort'] ) ? sanitize_key( $_GET['hjb_sort'] ) : 'newest';
		// phpcs:enable WordPress.Security.NonceVerification.Recommended

		$args = array(
			'post_type'      => 'humera_job_listing',
			'posts_per_page' => intval( $atts['posts_per_page'] ),
			'post_status'    => 'publish',
			'paged'          => $paged,
		);

		// Sorting
		if ( 'oldest' === $sort ) {
			$args['order'] = 'ASC';
		} elseif ( 'salary' === $sort ) {
			$args['meta_key'] = '_humera_job_salary'; // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
			$args['orderby']  = 'meta_value_num';
			$args['order']    = 'DESC';
		} else {
			$args['orderby'] = 'date';
			$args['order']   = 'DESC';
		}

		// phpcs:disable WordPress.Security.NonceVerification.Recommended -- Public search form
		if ( ! empty( $_GET['search_keywords'] ) ) {
			$args['s'] = sanitize_text_field( wp_unslash( $_GET['search_keywords'] ) );
		}
		if ( ! empty( $_GET['search_location'] ) ) {
			$args['meta_query'][] = array(
				'key'     => '_humera_job_location',
				'value'   => sanitize_text_field( wp_unslash( $_GET['search_location'] ) ),
				'compare' => 'LIKE',
			);
		}
		if ( ! empty( $_GET['search_category'] ) ) {
			$args['tax_query'][] = array(
				'taxonomy' => 'humera_job_category',
				'field'    => 'slug',
				'terms'    => sanitize_text_field( wp_unslash( $_GET['search_category'] ) ),
			);
		}
		if ( ! empty( $_GET['search_type'] ) ) {
			$args['tax_query'][] = array(
				'taxonomy' => 'humera_job_type',
				'field'    => 'slug',
				'terms'    => sanitize_text_field( wp_unslash( $_GET['search_type'] ) ),
			);
		}
		// phpcs:enable WordPress.Security.NonceVerification.Recommended

		$query = new WP_Query( $args );

		ob_start();

		// ── Hero Search Section ──────────────────────
		echo '<div class="humera-search-wrapper">';
		echo '<div class="humera-search-hero">';
		echo '<h2>' . esc_html__( 'Find Your Next Opportunity', 'humera-job-board' ) . '</h2>';
		echo '<p>' . esc_html__( 'Search thousands of jobs and find the perfect match.', 'humera-job-board' ) . '</p>';
		// Search form is built internally with all values escaped — no user input passes through unescaped.
		echo $this->render_search_form(); // phpcs:ignore WordPress.Security.EscapeOutput.OutputNotEscaped
		echo '</div>'; // .humera-search-hero
		echo '</div>'; // .humera-search-wrapper

		// Layout class for the job listings wrapper.
		$layout_class = ( 'list' === sanitize_key( $atts['layout'] ) ) ? 'humera-job-listings--list' : 'humera-job-listings--grid';

		// ── Job Listings ─────────────────────────────
		if ( $query->have_posts() ) {
			echo '<div class="humera-job-listings ' . esc_attr( $layout_class ) . '" id="humera-job-listings">';
			while ( $query->have_posts() ) {
				$query->the_post();
				$this->load_template( 'content-job-listing' );
			}
			echo '</div>';

			// Pagination
			echo '<div class="humera-pagination">';
			echo wp_kses_post( paginate_links( array(
				'total' => $query->max_num_pages,
			) ) );
			echo '</div>';

			wp_reset_postdata();
		} else {
			echo '<p>' . esc_html__( 'No jobs found. Try adjusting your search.', 'humera-job-board' ) . '</p>';
		}

		return ob_get_clean();
	}

	private function render_search_form() {
		// phpcs:disable WordPress.Security.NonceVerification.Recommended -- Public search form
		$keywords = isset( $_GET['search_keywords'] ) ? sanitize_text_field( wp_unslash( $_GET['search_keywords'] ) ) : '';
		$location = isset( $_GET['search_location'] ) ? sanitize_text_field( wp_unslash( $_GET['search_location'] ) ) : '';
		$category = isset( $_GET['search_category'] ) ? sanitize_text_field( wp_unslash( $_GET['search_category'] ) ) : '';
		$type     = isset( $_GET['search_type'] ) ? sanitize_text_field( wp_unslash( $_GET['search_type'] ) ) : '';
		// phpcs:enable WordPress.Security.NonceVerification.Recommended

		$categories = get_terms( array( 'taxonomy' => 'humera_job_category', 'hide_empty' => false ) );
		$types      = get_terms( array( 'taxonomy' => 'humera_job_type', 'hide_empty' => false ) );

		ob_start();
		?>
		<form method="GET" class="humera-job-search-form">
			<input type="text" name="search_keywords" placeholder="<?php esc_attr_e( 'Keywords', 'humera-job-board' ); ?>" value="<?php echo esc_attr( $keywords ); ?>">
			<input type="text" name="search_location" placeholder="<?php esc_attr_e( 'Location', 'humera-job-board' ); ?>" value="<?php echo esc_attr( $location ); ?>">
			
			<select name="search_category">
				<option value=""><?php esc_html_e( 'All Categories', 'humera-job-board' ); ?></option>
				<?php foreach ( $categories as $cat ) : ?>
					<option value="<?php echo esc_attr( $cat->slug ); ?>" <?php selected( $category, $cat->slug ); ?>><?php echo esc_html( $cat->name ); ?></option>
				<?php endforeach; ?>
			</select>

			<select name="search_type">
				<option value=""><?php esc_html_e( 'All Job Types', 'humera-job-board' ); ?></option>
				<?php foreach ( $types as $t ) : ?>
					<option value="<?php echo esc_attr( $t->slug ); ?>" <?php selected( $type, $t->slug ); ?>><?php echo esc_html( $t->name ); ?></option>
				<?php endforeach; ?>
			</select>

			<button type="submit"><?php esc_html_e( 'Search', 'humera-job-board' ); ?></button>
		</form>
		<?php
		return ob_get_clean();
	}

	public function load_single_template( $template ) {
		if ( is_singular( 'humera_job_listing' ) ) {
			// Prevent page-level caching so the nonce in the application form stays fresh.
			if ( ! defined( 'LSCACHE_NO_CACHE' ) ) {
				define( 'LSCACHE_NO_CACHE', true ); // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedConstantFound -- LiteSpeed Cache constant.
			}
			if ( ! defined( 'DONOTCACHEPAGE' ) ) {
				define( 'DONOTCACHEPAGE', true ); // phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedConstantFound -- Standard cache-prevention constant.
			}
			nocache_headers();

			// Increment View Count
			$job_id = get_the_ID();
			$views = get_post_meta( $job_id, '_humera_job_views', true );
			$views = $views ? intval( $views ) + 1 : 1;
			update_post_meta( $job_id, '_humera_job_views', $views );

			$plugin_template = HUMERA_JOB_BOARD_PATH . 'templates/single-job.php';
			if ( file_exists( $plugin_template ) ) {
				return $plugin_template;
			}
		}
		return $template;
	}

	public static function get_application_fields( $job_id ) {
		// Try to fetch from Form Editor options
		$saved_fields = get_option( 'humera_job_form_settings' );
		
		if ( ! empty( $saved_fields ) && is_array( $saved_fields ) ) {
			$fields = $saved_fields;
		} else {
			// Defaults
			$fields = array(
				'applicant_name' => array(
					'label'       => __( 'Full Name', 'humera-job-board' ),
					'type'        => 'text',
					'required'    => true,
					'placeholder' => '',
				),
				'applicant_email' => array(
					'label'       => __( 'Email Address', 'humera-job-board' ),
					'type'        => 'email',
					'required'    => true,
					'placeholder' => '',
				),
				'applicant_message' => array(
					'label'       => __( 'Cover Letter / Message', 'humera-job-board' ),
					'type'        => 'textarea',
					'required'    => false,
					'placeholder' => '',
				),
				'applicant_cv' => array(
					'label'       => __( 'Upload CV/Resume (PDF, Doc)', 'humera-job-board' ),
					'type'        => 'file',
					'required'    => true,
					'accept'      => '.pdf,.doc,.docx',
				),
			);
		}

		return apply_filters( 'humera_job_application_fields', $fields, $job_id );
	}

	public function handle_application_submission() {
		// Spam Protection (Honeypot)
		// Nonce verification not required for honeypot check, as it's a simple check for an empty field.
		if ( ! empty( $_POST['humera_honeypot'] ) ) { // phpcs:ignore WordPress.Security.NonceVerification.Missing
			wp_die( esc_html__( 'Spam detected.', 'humera-job-board' ) );
		}

		// Nonce verification is performed here.
		if ( ! isset( $_POST['humera_application_nonce'] ) || ! wp_verify_nonce( sanitize_text_field( wp_unslash( $_POST['humera_application_nonce'] ) ), 'humera_application_submit' ) ) { // phpcs:ignore WordPress.Security.ValidatedSanitizedInput, WordPress.Security.NonceVerification.Missing
			$job_id_fallback = isset( $_POST['job_id'] ) ? intval( $_POST['job_id'] ) : 0; // phpcs:ignore WordPress.Security.NonceVerification.Missing
			$back_url = $job_id_fallback ? get_permalink( $job_id_fallback ) : home_url();
			wp_die(
				esc_html__( 'Security check failed. This usually happens if the page was cached for too long. Please go back and try again.', 'humera-job-board' )
				. '<br><br><a href="' . esc_url( $back_url ) . '">' . esc_html__( '← Go back and retry', 'humera-job-board' ) . '</a>',
				esc_html__( 'Security Error', 'humera-job-board' ),
				array( 'response' => 403, 'back_link' => true )
			);
		}

		$job_id = isset( $_POST['job_id'] ) ? intval( $_POST['job_id'] ) : 0;
		if ( ! $job_id ) {
			wp_die( esc_html__( 'Invalid Job ID.', 'humera-job-board' ) );
		}

		$fields = self::get_application_fields( $job_id );
		$data = array();
		$cv_url = '';

		// Validate & Sanitize
		foreach ( $fields as $key => $field ) {
			if ( ! empty( $field['required'] ) ) {
				if ( $field['type'] === 'file' ) {
					if ( empty( $_FILES[ $key ]['name'] ) ) {
						/* translators: %s: Field Label */
						wp_die( sprintf( esc_html__( '%s is required.', 'humera-job-board' ), esc_html( $field['label'] ) ) );
					}
				} else {
					if ( empty( $_POST[ $key ] ) ) {
						/* translators: %s: Field Label */
						wp_die( sprintf( esc_html__( '%s is required.', 'humera-job-board' ), esc_html( $field['label'] ) ) );
					}
				}
			}

			if ( $field['type'] === 'text' || $field['type'] === 'select' ) {
				$data[ $key ] = isset( $_POST[ $key ] ) ? sanitize_text_field( wp_unslash( $_POST[ $key ] ) ) : '';
			} elseif ( $field['type'] === 'email' ) {
				$data[ $key ] = isset( $_POST[ $key ] ) ? sanitize_email( wp_unslash( $_POST[ $key ] ) ) : '';
			} elseif ( $field['type'] === 'textarea' ) {
				$data[ $key ] = isset( $_POST[ $key ] ) ? sanitize_textarea_field( wp_unslash( $_POST[ $key ] ) ) : '';
			} elseif ( $field['type'] === 'file' && ! empty( $_FILES[ $key ]['name'] ) ) {
				// Handle File Upload - currently optimized for 'applicant_cv' but could be genericized
				require_once( ABSPATH . 'wp-admin/includes/file.php' );
				// Sanitize file array structure - though wp_handle_upload handles the file itself, we should ensure the key exists safely.
				if ( isset( $_FILES[ $key ] ) ) {
					// phpcs:ignore WordPress.Security.ValidatedSanitizedInput.InputNotSanitized
					$uploadedfile = $_FILES[ $key ];
					// Sanitize the file array before passing to wp_handle_upload
					$uploadedfile = array_map( 'sanitize_text_field', wp_unslash( $uploadedfile ) );

					$upload_overrides = array( 'test_form' => false );
					$movefile = wp_handle_upload( $uploadedfile, $upload_overrides );
	
					if ( $movefile && ! isset( $movefile['error'] ) ) {
						$url = $movefile['url'];
						if ( $key === 'applicant_cv' ) {
							$cv_url = $url;
						}
						$data[ $key ] = $url;
					} else {
						wp_die( esc_html( $movefile['error'] ) );
					}
				}
			}
		}

		$name    = isset( $data['applicant_name'] ) ? $data['applicant_name'] : __( 'Anonymous', 'humera-job-board' );
		$email   = isset( $data['applicant_email'] ) ? $data['applicant_email'] : '';
		$message = isset( $data['applicant_message'] ) ? $data['applicant_message'] : '';

		// Create Application Post.
		// Temporarily switch to an admin user so wp_insert_post() has full capabilities.
		$original_user_id = get_current_user_id();
		$admin_users      = get_users( array( 'role' => 'administrator', 'number' => 1, 'fields' => 'ID' ) );
		$admin_id         = ! empty( $admin_users ) ? (int) $admin_users[0] : 1;

		wp_set_current_user( $admin_id );

		/* translators: 1: Job Title, 2: Applicant Name */
		$post_title = sprintf( __( 'Application for %1$s by %2$s', 'humera-job-board' ), get_the_title( $job_id ), $name );

		$application_id = wp_insert_post(
			array(
				'post_title'   => $post_title,
				'post_content' => $message,
				'post_status'  => 'publish',
				'post_type'    => 'humera_app',
				'post_author'  => $admin_id,
			),
			true
		);

		// Restore original user context.
		wp_set_current_user( $original_user_id );

		if ( is_wp_error( $application_id ) ) {
			$back_url = get_permalink( $job_id );
			wp_die(
				esc_html__( 'Error saving application. Please go back and try again.', 'humera-job-board' )
				. '<br><br><a href="' . esc_url( $back_url ) . '">' . esc_html__( '← Go back and retry', 'humera-job-board' ) . '</a>',
				esc_html__( 'Application Error', 'humera-job-board' ),
				array( 'response' => 500, 'back_link' => true )
			);
		}

		update_post_meta( $application_id, '_job_applied_for', $job_id );
		update_post_meta( $application_id, '_applicant_name', $name );
		update_post_meta( $application_id, '_applicant_email', $email );
		if ( $cv_url ) {
			update_post_meta( $application_id, '_applicant_cv', $cv_url );
		}
		
		// Save full data array for custom fields
		update_post_meta( $application_id, '_humera_application_data', $data );

		// Send Email Notification
		$to = get_post_meta( $job_id, '_humera_job_application_email', true );
		if ( ! $to ) {
			$to = get_option( 'admin_email' );
		}
		
		$admin_email = get_option( 'admin_email' );
		$headers = array( 'Content-Type: text/plain; charset=UTF-8' );
		
		// Add BCC to Admin
		if ( $to !== $admin_email ) {
			$headers[] = 'Bcc: ' . $admin_email;
		}

		/* translators: %s: Job Title */
		$subject = sprintf( __( 'New Application: %s', 'humera-job-board' ), get_the_title( $job_id ) );
		
		/* translators: %s: Job Title */
		$body = sprintf( __( "New application received for: %s\n\n", 'humera-job-board' ), get_the_title( $job_id ) );
		foreach ( $fields as $key => $field ) {
			if ( $field['type'] !== 'file' && isset( $data[ $key ] ) ) {
				$body .= $field['label'] . ': ' . $data[ $key ] . "\n";
			} elseif ( $field['type'] === 'file' && isset( $data[ $key ] ) ) {
				$body .= $field['label'] . ': ' . $data[ $key ] . "\n";
			}
		}

		wp_mail( $to, $subject, $body, $headers );

		// Redirect back to job page with success message
		wp_safe_redirect( add_query_arg( 'application_sent', 'true', get_permalink( $job_id ) ) );
		exit;
	}

	private function load_template( $template_name ) {
		$template_path = HUMERA_JOB_BOARD_PATH . 'templates/' . $template_name . '.php';
		if ( file_exists( $template_path ) ) {
			include $template_path;
		} else {
			// Fallback content if template missing
			echo '<div class="humera-job-item">';
			echo '<h3><a href="' . esc_url( get_permalink() ) . '">' . esc_html( get_the_title() ) . '</a></h3>';
			echo '<div class="humera-job-meta">';
			echo '<span>' . esc_html( get_post_meta( get_the_ID(), '_humera_job_location', true ) ) . '</span>';
			echo ' | ';
			echo '<span>' . get_the_term_list( get_the_ID(), 'humera_job_type', '', ', ' ) . '</span>'; // Term list contains HTML, needed for links
			echo '</div>';
			echo '</div>';
		}
	}

}
